/*
 * past.h: This file is part of the PAST project.
 *
 * PAST: the PoCC Abstract Syntax Tree
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#ifndef PAST_PAST_H
# define PAST_PAST_H

# include <past/common.h>
# include <past/symbols.h>

BEGIN_C_DECLS


/**
 * Type the nodes based on their free function (declared as a global).
 *
 */
/** Explicitely set a hierarchy with at most 5 levels. **/
#define PAST_NODE_HIERARCHY_HEIGHT 5

struct past_node_t;
struct past_node_type_t
{
  void (*freefun[PAST_NODE_HIERARCHY_HEIGHT])(struct past_node_t*);
};
typedef const struct past_node_type_t cs_past_node_type_t;
typedef void (*past_fun_t) (struct past_node_t* node, void* data);
typedef struct past_node_t* (*past_clone_fun_t) (struct past_node_t* node);
typedef void (*past_visitor_fun_t) (struct past_node_t* root,
				    past_fun_t prefix,
				    void* prefix_data,
				    past_fun_t suffix,
				    void* suffix_data);
struct past_node_t {
  cs_past_node_type_t*	type; // Unique id for the node type (eg, past_root).
  past_visitor_fun_t	visitor; // pointer to the visitor function.
  past_clone_fun_t	clone; // pointer to the clone function.
  struct past_node_t*	parent; // Parent pointer.
  struct past_node_t*	next; // Chained list of siblings.
  void*			metainfo; // Managed (ie, freed) user pointer.
  void*			usr; // Unmanaged user pointer.
};
typedef struct past_node_t s_past_node_t;


/**
 * Convenience macros. To register a node of type 'mytype':
 * - in the header (.h) file, use PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(mytype)
 * - in the unit (.c) file, use PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(mytype)
 *   after declaring the free function associated to 'type', which
 *   MUST be declared 'static void past_mytype_free(s_past_node_t*)'.
 *   Similarely, the generic visitor MUST be declared 'static void
 *   past_mytype_visitor(s_past_node_t*)'.
 *
 * if 'type' is a subclass of a super class 'super', it should be
 * registered in the unit file with
 * PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_2(type, super)
 *
 */
#define PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(type)	\
extern cs_past_node_type_t* const past_##type;
#define PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(type)			\
cs_past_node_type_t s_past_##type = { past_##type##_free, NULL };	\
cs_past_node_type_t* const past_##type = &s_past_##type;
#define PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_2(type,superclass)	       	\
cs_past_node_type_t s_past_##type = { past_##type##_free, past_##superclass##_free, NULL }; \
cs_past_node_type_t* const past_##type = &s_past_##type;
#define PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_3(type,super1,super2)     	\
cs_past_node_type_t s_past_##type = { past_##type##_free, past_##super1##_free, past_##super2##_free, NULL }; \
cs_past_node_type_t* const past_##type = &s_past_##type;

/**
 *
 *
 */
struct past_root_t
{
  s_past_node_t		node;
  //
  s_symbol_table_t*    	symboltable;
  s_past_node_t*	body;
};
typedef struct past_root_t s_past_root_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(root);

/**
 *
 *
 */
struct past_variable_t
{
  s_past_node_t		node;
  //
  s_symbol_t*  		symbol;
  // User field, not maintained.
  void*			usr;
};
typedef struct past_variable_t s_past_variable_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(variable);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(expr);


/**
 *
 *
 */
enum e_past_value_type
{
  e_past_value_unknown = 0,
  e_past_value_bool = 1,
  e_past_value_char = 2,
  e_past_value_uchar = 3,
  e_past_value_int = 4,
  e_past_value_uint = 5,
  e_past_value_longint = 6,
  e_past_value_ulongint = 7,
  e_past_value_longlongint = 8,
  e_past_value_ulonglongint = 9,
  e_past_value_float = 10,
  e_past_value_double = 11,
  e_past_value_longdouble = 12,
  e_past_value_longlong = 13,
  e_past_value_ptr = 14
};
typedef enum e_past_value_type e_past_value_type_t;

union u_past_value_data
{
  char boolval;
  char charval;
  unsigned char ucharval;
  int intval;
  unsigned int uintval;
  long int longintval;
  unsigned long int ulongintval;
  long long int longlongintval;
  unsigned long long int ulonglongintval;
  float floatval;
  double doubleval;
  long double longdoubleval;
  long long longlongval;
  void* ptrval;
};
typedef union u_past_value_data u_past_value_data_t;

struct past_value_t
{
  s_past_node_t		node;
  //
  int			type; // by default, should be of e_past_value_type_t.
  u_past_value_data_t	value;// Use ptrval for user-defined types.
};
typedef struct past_value_t s_past_value_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(value);


/**
 *
 *
 */
struct past_binary_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	lhs;
  s_past_node_t*	rhs;
};
typedef struct past_binary_t s_past_binary_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(binary);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(add);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(sub);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(mul);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(div);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(mod);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(min);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(max);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(ceild);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(floord);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(and);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(or);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(equal);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(assign);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(geq);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(leq);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(gt);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(lt);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(arrayref);

/**
 *
 *
 */
struct past_unary_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	expr;
};
typedef struct past_unary_t s_past_unary_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(unary);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(round);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(floor);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(ceil);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(sqrt);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(inc_before);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(inc_after);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(dec_before);
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(dec_after);

/**
 * Loop types.
 *
 */
enum e_past_loop_type
{
  e_past_unknown_loop = 0,
  e_past_tile_loop = 1,
  e_past_point_loop = 2,
  e_past_vector_loop = 3,
  e_past_openmp_loop = 4,
  e_past_otl_loop = 5,
  e_past_fulltile_loop = 6,
};
typedef enum e_past_loop_type e_past_loop_type_t;

/**
 * past_for
 *
 */
struct past_for_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	init;
  s_past_node_t*	test;
  s_past_variable_t*	iterator;
  s_past_node_t*	increment;
  s_past_node_t*	body;
  int			type; // Should be of type e_past_loop_type_t
  void*			usr;
};
typedef struct past_for_t s_past_for_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(for);

/**
 * past_for <- past_parfor
 *
 */
struct past_parfor_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	init;
  s_past_node_t*	test;
  s_past_variable_t*	iterator;
  s_past_node_t*	increment;
  s_past_node_t*	body;
  int			type; // Should be of type e_past_loop_type_t
  void*			usr;
  s_past_variable_t**	private_vars;
};
typedef struct past_parfor_t s_past_parfor_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(parfor);


/**
 *
 *
 */
struct past_block_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	body;
};
typedef struct past_block_t s_past_block_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(block);



/**
 *
 *
 */
struct past_if_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	condition;
  s_past_node_t*	then_clause;
  s_past_node_t*	else_clause;
};
typedef struct past_if_t s_past_if_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(if);


/**
 *
 *
 */
struct past_affineguard_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	condition;
  s_past_node_t*	then_clause;
};
typedef struct past_affineguard_t s_past_affineguard_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(affineguard);


/**
 *
 *
 */
struct past_cloogstmt_t
{
  s_past_node_t		node;
  //
  void*			cloogdomain; // CloogDomain
  void*			cloogstatement; // CloogStatement
  s_past_variable_t*	stmt_name;
  int			stmt_number;
  s_past_node_t*	substitutions; // list of iterator substitutions
  s_past_node_t*	references; // chained list of access function trees.
};
typedef struct past_cloogstmt_t s_past_cloogstmt_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(cloogstmt);


/**
 *
 *
 */
struct past_statement_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	body;
};
typedef struct past_statement_t s_past_statement_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(statement);


/**
 * Function call.
 *
 */
struct past_funcall_t
{
  s_past_node_t		node;
  //
  s_past_variable_t*  	name;
  s_past_node_t*	args_list;
};
typedef struct past_funcall_t s_past_funcall_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(funcall);

/**
 * Variable declaration (will print 'type var').
 *
 */
struct past_vardecl_t
{
  s_past_node_t		node;
  //
  s_past_variable_t*  	name;
  s_past_variable_t*  	type;
};
typedef struct past_vardecl_t s_past_vardecl_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(vardecl);


/**
 *
 *
 */
struct past_ternary_cond_t
{
  s_past_node_t		node;
  //
  s_past_node_t*	cond;
  s_past_node_t*	true_clause;
  s_past_node_t*	false_clause;
};
typedef struct past_ternary_cond_t s_past_ternary_cond_t;
PAST_DECLARE_NODE_IN_HIERARCHY_HEADER(ternary_cond);



/******************************************************************************/
/******************************* Node creation ********************************/
/******************************************************************************/

extern
s_past_root_t* past_root_create (s_symbol_table_t* symboltable,
				 s_past_node_t* body);
extern
s_past_node_t* past_node_root_create (s_symbol_table_t* symboltable,
				      s_past_node_t* body);

extern
s_past_variable_t* past_variable_create (s_symbol_t* symbol);
extern
s_past_node_t* past_node_variable_create (s_symbol_t* symbol);

extern
s_past_value_t* past_value_create (int type, u_past_value_data_t val);
extern
s_past_node_t* past_node_value_create (int type, u_past_value_data_t val);
extern
s_past_value_t* past_value_create_from_int (int val);
extern
s_past_node_t* past_node_value_create_from_int (int val);

extern
s_past_for_t* past_for_create (s_past_node_t* init,
			       s_past_node_t* test,
			       s_past_variable_t* iterator,
			       s_past_node_t* increment,
			       s_past_node_t* body,
			       void* usr);
extern
s_past_node_t* past_node_for_create (s_past_node_t* init,
				     s_past_node_t* test,
				     s_past_variable_t* iterator,
				     s_past_node_t* increment,
				     s_past_node_t* body,
				     void* usr);

extern
s_past_parfor_t* past_parfor_create (s_past_node_t* init,
				     s_past_node_t* test,
				     s_past_variable_t* iterator,
				     s_past_node_t* increment,
				     s_past_node_t* body,
				     void* usr);
extern
s_past_node_t* past_node_parfor_create (s_past_node_t* init,
					s_past_node_t* test,
					s_past_variable_t* iterator,
					s_past_node_t* increment,
					s_past_node_t* body,
					void* usr);

extern
s_past_binary_t* past_binary_create (cs_past_node_type_t* type,
				     s_past_node_t* lhs,
				     s_past_node_t* rhs);
extern
s_past_node_t* past_node_binary_create (cs_past_node_type_t* type,
					s_past_node_t* lhs,
					s_past_node_t* rhs);

extern
s_past_unary_t* past_unary_create (cs_past_node_type_t* type,
				   s_past_node_t* expr);
extern
s_past_node_t* past_node_unary_create (cs_past_node_type_t* type,
				       s_past_node_t* expr);

extern
s_past_block_t* past_block_create (s_past_node_t* body);
extern
s_past_node_t* past_node_block_create (s_past_node_t* body);

extern
s_past_if_t* past_if_create (s_past_node_t* cond,
			     s_past_node_t* then_clause,
			     s_past_node_t* else_clause);
extern
s_past_node_t* past_node_if_create (s_past_node_t* cond,
				    s_past_node_t* then_clause,
				    s_past_node_t* else_clause);

extern
s_past_affineguard_t* past_affineguard_create (s_past_node_t* cond,
					       s_past_node_t* then_clause);
extern
s_past_node_t* past_node_affineguard_create (s_past_node_t* cond,
					     s_past_node_t* then_clause);

extern
s_past_cloogstmt_t* past_cloogstmt_create (void* cloogdomain,
					   void* cloogstatement,
					   s_past_variable_t* stmt_name,
					   int stmt_number,
					   s_past_node_t* substitutions);
extern
s_past_node_t* past_node_cloogstmt_create (void* cloogdomain,
					   void* cloogstatement,
					   s_past_variable_t* stmt_name,
					   int stmt_number,
					   s_past_node_t* substitutions);

extern
s_past_statement_t* past_statement_create (s_past_node_t* body);
extern
s_past_node_t* past_node_statement_create (s_past_node_t* body);

extern
s_past_funcall_t* past_funcall_create (s_past_variable_t* name,
				       s_past_node_t* args_list);
extern
s_past_node_t* past_node_funcall_create (s_past_variable_t* name,
					 s_past_node_t* args_list);

extern
s_past_vardecl_t* past_vardecl_create (s_past_variable_t* name,
				       s_past_variable_t* type);
extern
s_past_node_t* past_node_vardecl_create (s_past_variable_t* name,
					 s_past_variable_t* type);

extern
s_past_ternary_cond_t* past_ternary_cond_create (s_past_node_t* cond,
						 s_past_node_t* true_clause,
						 s_past_node_t* false_clause);
extern
s_past_node_t* past_node_ternary_cond_create (s_past_node_t* cond,
						 s_past_node_t* true_clause,
						 s_past_node_t* false_clause);

/******************************************************************************/
/****************************** Node processing *******************************/
/******************************************************************************/

/**
 * Check the type of a node.
 *
 */
extern
int
past_node_is_a (s_past_node_t* node, cs_past_node_type_t* type);
#define PAST_NODE_IS_A(node, type) past_node_is_a(node, type)
#define PAST_DECLARE_TYPED(type, typednode, node)      	\
  s_past_##type##_t* typednode = (s_past_##type##_t*) node


/**
 * Deep copy.
 *
 */
extern
s_past_node_t* past_clone (s_past_node_t* node);

/**
 * Shallow copy.
 *
 */
extern
s_past_node_t* past_copy (s_past_node_t* node);

/**
 * Deep free.
 *
 */
extern
void past_deep_free (s_past_node_t* node);

/**
 * Sets the parent pointer in a tree.
 *
 */
extern
void past_set_parent (s_past_node_t* root);

/**
 * Generic visitor.
 *
 */
extern
void past_visitor (s_past_node_t* node,
		   past_fun_t prefix,
		   void* prefix_data,
		   past_fun_t suffix,
		   void* suffix_data);

/**
 * Replace a node, assuming old->parent is set and old is not a past_root.
 *
 *
 */
extern
void past_replace_node (s_past_node_t* oldnode,
			s_past_node_t* newnode);

/**
 * Get the address of a node in the hierarchy.
 *
 *
 */
extern
s_past_node_t** past_node_get_addr (s_past_node_t* node);


/**
 * Convert a past_for node into a past_parfor node. If the node parent
 * is set, the new node is correctly inserted in the hierarchy.
 * Returns the newly created node, and deletes the current node.
 */
s_past_node_t*
past_for_to_parfor (s_past_node_t* node);


END_C_DECLS


#endif // PAST_PAST_H
