/*
 * fusionspace.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/fusionspace.h>
#include <letsee/lspace.h>


#undef echo
#define echo(S) { printf(#S); printf("\n"); }


static
int
ls_fusionspace_check_system (CandlProgram* program,
			     s_ls_options_t* options,
			     CandlDependence* dep,
			     int type,
			     s_fm_compsol_t* csF)
{
  s_fm_system_t* to_solve;
  s_fm_compsol_t* cs_tosolve;
  s_fm_solution_t* solution;
  s_fm_solution_t* solfull;
  int dimension;
  int i, j;
  int ret = 1; // Returns 1 if this dependence is fusion-preventing.

  Z_ASSIGN_SI(options->lb, 0);
  Z_ASSIGN_SI(options->Ub, 1000);
/*   Z_ASSIGN_SI(options->plb, 0); */
/*   Z_ASSIGN_SI(options->pUb, 1000); */
/*   Z_ASSIGN_SI(options->clb, 0); */
/*   Z_ASSIGN_SI(options->cUb, 1000); */

  // Build the initial system.
  to_solve = ls_farkas_build_system (program, options, dep,
				     &dimension, type);
  // Permute the columns: put the parameter columns at the beginning.
  int nb_param = program->context->NbColumns - 2;
  int nb_cst = 1;
  int iter_p = dep->source->depth;
  if (dep->source != dep->target)
    {
      iter_p += dep->target->depth;
      nb_param *= 2;
      ++nb_cst;
    }
  nb_param += nb_cst;
  for (i = 0; i < nb_param; ++i)
    for (j = iter_p + 1 + i; j > 1 + i; --j)
      fm_system_swap_column (to_solve, j, j - 1);
  // Project the Farkas multipliers.
  cs_tosolve = fm_compsol_init_sys (to_solve);
  fm_system_free (to_solve);
  to_solve = fm_solution_to_system (cs_tosolve->poly);
  fm_solution_free (cs_tosolve->poly);
  if (options->noredundancy_solver)
    cs_tosolve->poly = fm_solver
      (to_solve, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN);
  else
    cs_tosolve->poly = fm_solver
      (to_solve, FM_SOLVER_FAST);
  solution = fm_compsol_expand (cs_tosolve);
  solfull = fm_solution_dup (solution);
  fm_solution_cut (solution, nb_param);
  fm_compsol_free (cs_tosolve);
  // Check if there's a solution with all parametric shift
  // coefficients = 0.
  s_fm_vector_t* v = fm_vector_alloc (nb_param + 1);
  ret = ! fm_solution_point_included (solution, v);
  fm_vector_free (v);
  if (! ret)
    {
      // Check this solution is not \vec{0}.
      s_fm_system_t* s = fm_solution_to_system (solfull);
      z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
      for (i = 0; i < nb_param; ++i)
	{
	  v = fm_vector_alloc (s->nb_cols);
	  fm_vector_assign_int_idx (v, one, i + 1);
	  fm_system_add_line (s, v);
	}
      if (fm_piptools_check_int (s))
	{
	  s_fm_compsol_t* cs = fm_compsol_init_sys (s);
	  s_fm_solution_t* sol = fm_compsol_expand (cs);
	  s_fm_rational_t* lb = NULL;
	  s_fm_rational_t* ub = NULL;
	  v = fm_vector_alloc (s->nb_cols);
	  for (i = 0; i < sol->size; ++i)
	    {
	      fm_solver_compute_min (&lb,
				     sol->solution[i].positive,
				     v, i, FM_MINLEXICO_INT);
	      fm_solver_compute_max (&ub,
				     sol->solution[i].negative,
				     v, i, FM_MINLEXICO_INT);
	      if (fm_rational_cmp (lb, ub) != 0 || Z_CMP_SI(lb->num, !=, 0))
		{
		  printf ("at %d: %d %d\n", i + 1, lb->num, ub->num);
		  break;
		}
	      else
		fm_vector_assign_idx (v, lb, i + 1);
	    }
	  if (i == sol->size)
	    ret = 1;
	  fm_rational_free (lb);
	  fm_rational_free (ub);
	  fm_solution_free (sol);
	  fm_compsol_free (cs);
	  fm_vector_free (v);
	}
      else
	ret = 1;
      fm_system_free (s);
      Z_CLEAR(one);
    }
  // Be clean.
  fm_solution_free (solution);
  fm_solution_free (solfull);
  // The constraints enforced by the dependence are requested.
  if (ret)
    {
      cs_tosolve = fm_compsol_init_sys (to_solve);
      fm_system_free (to_solve);
      to_solve = fm_solution_to_system (cs_tosolve->poly);
      fm_solution_free (cs_tosolve->poly);
      if (options->noredundancy_solver)
	cs_tosolve->poly = fm_solver
	  (to_solve, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN);
      else
	cs_tosolve->poly = fm_solver
	  (to_solve, FM_SOLVER_FAST);
      solution = fm_compsol_expand (cs_tosolve);
      fm_solution_cut (solution, dimension);
      fm_compsol_free (cs_tosolve);
      ls_farkas_intersect_cs (program, options, dep, csF, solution);
      fm_solution_free (solution);
    }

  Z_ASSIGN_SI(options->lb, -1000);
  Z_ASSIGN_SI(options->Ub, 1000);

  return ret;
}


static
s_ls_space_t*
ls_fusionspace_compute_solution (s_graph_t* graph,
				 CandlProgram* program,
				 s_ls_options_t* options)
{


/*   PA[count] = NULL; */
/*   CA[count] = NULL; */

  s_vertex_t* v;
  int id = 0;
  s_fm_solution_t* P;
  s_fm_solution_t* F;
  CandlDependence* dep;
  int dimension;
  s_fm_system_t* to_solve;
  s_fm_compsol_t* cs_tosolve;
  s_fm_compsol_t* cs_tmp;
  s_fm_compsol_t* csP;
  s_fm_compsol_t* csF;
  s_fm_solution_t* solution;
  s_vertex_t* tmpv;
  int cur_time_dim = 0;
  int max_dim = 0;
  int i;
  int check;
  int unresolved = graph->root != NULL;

  // Compute maximum depth of the schedule (max_depth + 1);
  for (i = 0; i < program->nb_statements; ++i)
    max_dim = max_dim < program->statement[i]->depth ?
      program->statement[i]->depth : max_dim;
  // Array to store output polyhedra.
  s_fm_compsol_t** CF = XMALLOC(s_fm_compsol_t*, max_dim + 1);
  s_fm_solution_t** EF = XMALLOC(s_fm_solution_t*, max_dim + 1);
  CF[max_dim] = NULL;
  EF[max_dim] = NULL;

  // Relax the bounds on coefficients.
  Z_ASSIGN_SI(options->lb, 0);
  Z_ASSIGN_SI(options->Ub, 1);
  Z_ASSIGN_SI(options->plb, 0);
  Z_ASSIGN_SI(options->pUb, program->nb_statements);
  Z_ASSIGN_SI(options->clb, 0);
  Z_ASSIGN_SI(options->cUb, program->nb_statements);
  Z_ASSIGN_SI(options->cUb, 0);

  // Iterate while unresolved dependence remains or max schedule
  // dimension not attained.
  while (graph->root && cur_time_dim < max_dim + 1 && unresolved)
    {
      if (options->verbose)
	{
	  fprintf (options->out_file,
		   ".... Current time dimension: %d\n", cur_time_dim + 1);
	}

      // Initialize the search space: relax constraints on coefficients,
      // \delta\Theta \ge 0
      P = ls_farkas_build_fco_constraints (graph, program, options);

      // Initialize the compacted solution polytopes.
      if (options->verbose)
	fprintf (options->out_file, "..... Compute compacted sol\n");
      csP = fm_compsol_init_sol (P);
/*       csF = fm_compsol_dup (csP); */
      fm_solution_free (P);

      // Iterate on all dependences, to detect the set of fusion
      // preventing dependences.
      if (options->verbose)
	fprintf (options->out_file, "..... Iterate on dependences\n");
      unresolved = 0;
      for (v = graph->root; v; v = v->next)
	{
	  dep = v->data;
	  if (! CANDL_DEP_TAG(dep, solved))
	    {
	      // Project Farkas multipliers on schedule coefficients.
	      // \Delta\Theta \ge 1.
	      solution = ls_farkas_project_multipliers
		(program, options, dep, LS_FARKAS_GEONE);

	      // Backup the system.
	      cs_tmp = fm_compsol_dup (csP);

	      // Add the obtained solution set to the solution polytope.
	      if (solution)
		ls_farkas_intersect_cs (program, options, dep, csP, solution);
	      else
		csP->empty = 1;

	      // Be clean.
	      fm_solution_free (solution);

	      if (options->verbose)
		{
		  fprintf (options->out_file,
			   "..... Strong dependence satisfaction: ");
		  fflush (options->out_file);
		}
	      // Check for emptiness of S /\ W_dep
	      if (csP->empty ||
		  ! fm_piptools_check_sol (csP->poly, FM_PIPTOOLS_INT))
		{
		  // No point. We need to check fusion-preventing on weak
		  // satisfaction.
		  if (options->verbose)
		    fprintf (options->out_file, "no\n");
		  unresolved = 1;
/* 		  check = ls_fusionspace_check_system */
/* 		    (program, options, dep, LS_FARKAS_GEZERO, csF); */

		  fm_compsol_free (csP);
		  csP = cs_tmp;
		}
	      else
		{
		  // A point. We need to check fusion-preventing on strong
		  // satisfaction.
		  if (options->verbose)
		    fprintf (options->out_file, "yes\n");
		  // If the dependence needs to be strongly solved at this
		  // depth because it's fusion preventing, mark it as
		  // solved.
/* 		  check = dep->solved = ls_fusionspace_check_system */
/* 		    (program, options, dep, LS_FARKAS_GEZERO, csF); */
		  CANDL_DEP_TAG(dep, solved) = 1;
		}
/* 	      if (options->verbose) */
/* 		{ */
/* 		  if (check) */
/* 		    printf ("..... Dependence *IS* preventing fusion\n"); */
/* 		  else */
/* 		    printf ("..... Dependence is not preventing fusion\n"); */
/* 		} */

/* 	      // Restore the system. */
/* 	      fm_compsol_free (csP); */
/* 	      csP = cs_tmp; */

	    }
	}
/*       // Remove all solved dependences from the dependence graph. */
/*       for (v = graph->root; v; ) */
/* 	{ */
/* 	  dep = v->data; */
/* 	  tmpv = v->next; */
/* 	  if (dep->solved == 1) */
/* 	    ls_graph_remove_vertex (graph, v); */
/* 	  v = tmpv; */
/* 	} */

      // Store the computed polyhedra for this time dimension.
      EF[cur_time_dim] = fm_compsol_expand (csP);
      CF[cur_time_dim] = csP;
      // Increase time dimension.
      cur_time_dim++;
    }

/*   // Complex case are not handled yet. */
/*   assert (unresolved == 0); */

  // Allocate and fill the output.
  s_ls_space_t* space = ls_space_alloc ();
  space->polyhedron = NULL;
  space->u_polyhedron = EF;
  space->u_compacted = CF;
  space->dimension = cur_time_dim - 1;

  return space;
}


/**
 * Ensure there is at least one constraint + and - per variable in the space.
 *
 */
void
ls_fusionspace_normalize_space (CandlProgram* program,
				s_ls_options_t* options,
				s_ls_space_t* space,
				z_type_t lb,
				z_type_t Ub)
{
  int i, j;
  s_fm_list_t* l;
  s_fm_vector_t* v;
  s_fm_solution_t* s;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);

  for (i = 0; i < space->dimension; ++i)
    {
      s = space->u_polyhedron[i];
      for (j = 0; j < s->size; ++j)
	{
/* 	  if (s->solution[j].positive == NULL) */
	    {
	      v = fm_vector_alloc (j + 3);
	      fm_vector_assign_int_idx (v, one, j + 1);
	      fm_vector_assign_int_idx (v, lb, j + 2);
	      fm_vector_set_ineq (v);
	      fm_solution_add_line_at (s, v, j + 1);
	    }
/* 	  if (s->solution[j].negative == NULL) */
	    {
	      v = fm_vector_alloc (j + 3);
	      fm_vector_assign_int_idx (v, mone, j + 1);
	      fm_vector_assign_int_idx (v, Ub, j + 2);
	      fm_vector_set_ineq (v);
	      fm_solution_add_line_at (s, v, j + 1);
	    }
	}
    }
  Z_CLEAR(one);
  Z_CLEAR(mone);
}



/**
 * \brief Build the set of legal parametric fusions.
 *
 * The output solution is represented by the polytope array Pa,
 * organized as follows:
 * For s1 = [i1 p1 c1] the schedule of the first statement (i1 is the
 * "iterator" part of the schedule, p1 the parameter part and c1 is
 * the constant one), and respectivly s2 = [ i2 p2 c2 ] for the second
 * statement; the output polytope Pa[x] is of the form Pa[x] = [ p1 p2].
 * The process is the same for each dimension x of the
 * schedule.
 *
 */
s_ls_space_t*
ls_fusionspace_build (CandlProgram* program,
		      CandlDependence* dependences,
		      s_ls_options_t* options)
{
  int i, j, k;
  s_ls_space_t* space = NULL;
  s_graph_t* graph;
  CandlDependence* tmp;
  s_vertex_t* vertex;

  if (options->verbose)
    fprintf(options->out_file, "... Build and solve local systems\n");

  // Preprocess the dependence graph.
  CandlDependence* depgraph = // dependences;
    ls_schedspace_depgraph_preprocess (program, dependences);
  candl_dependence_pprint (stdout, depgraph);

  // Build the dependence graph from Candl's representation.
  graph = ls_graph_alloc ();
  int uid = 0;
  for (tmp = depgraph; tmp; tmp = tmp->next)
    {
      // Reset the 'solved' bit.
      CANDL_DEP_TAG(tmp, solved) = 0;
      vertex = ls_graph_create_vertex (graph, tmp, uid++);
    }

  // Compute the partitions (FCO only constraint)
  // Compute the solution space for each partitions
  // Merge the spaces


  // Compute the solution polytope.
  space = ls_fusionspace_compute_solution (graph, program, options);

  if (options->verbose)
    fprintf(options->out_file, "... All systems solved\n");

  // Initial PIP check.
  for (i = 0; i < space->size; ++i)
    if (! fm_piptools_check_sol (space->u_polyhedron[i], FM_PIPTOOLS_INT))
      {
	if (options->verbose)
	  fprintf(options->out_file,
		  "=> Ending computation: no point at dimension %d\n",
		  i + 1);
	exit (1);
      }

  // Compute maximum depth of the schedule (max_depth + 1);
  int max_dim = 0;
  for (i = 0; i < program->nb_statements; ++i)
    max_dim = max_dim < program->statement[i]->depth ?
      program->statement[i]->depth : max_dim;

  // If some time dimensions where not computed, we have to fill them.
  s_fm_solution_t* P;
  s_fm_compsol_t* cs;
  if (space->dimension < max_dim)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "... Fill remaining time dimension(s): %d/%d\n",
		 max_dim - space->dimension, max_dim);

      P = ls_farkas_build_fco_constraints (graph, program, options);
      cs = fm_compsol_init_sol (P);
      fm_solution_free (P);
      for (i = space->dimension; i < max_dim; ++i)
	{
	  space->u_compacted[i] = fm_compsol_dup (cs);
	  space->u_polyhedron[i] = fm_compsol_expand (cs);
	}
      fm_compsol_free (cs);
      space->dimension = max_dim;
    }

  // Permute the time dimensions.
  if (options->verbose)
    fprintf (options->out_file, "... Permute time dimension(s)\n");
  for (i = 0; i < space->dimension / 2; ++i)
    {
      P = space->u_polyhedron[i];
      cs = space->u_compacted[i];
      space->u_polyhedron[i] = space->u_polyhedron[space->dimension - i - 1];
      space->u_compacted[i] = space->u_compacted[space->dimension - i - 1];
      space->u_polyhedron[space->dimension - i - 1] = P;
      space->u_compacted[space->dimension - i - 1] = cs;
    }


  // Permute the columns associated to parameters, and restore the
  // space structure.
  if (options->verbose)
    fprintf (options->out_file,
	     "... Put parameter columns at the beginning\n");
  int nb_param = 0;
  int iter_p = 0;
  for (i = 0; i < program->nb_statements; ++i)
    {
      iter_p += program->statement[i]->depth;
      nb_param += program->context->NbColumns - 2;
    }
  for (k = 0; k < space->dimension; ++k)
    {
      s_fm_system_t* s = fm_solution_to_system (space->u_polyhedron[k]);
      for (i = 0; i < nb_param; ++i)
	for (j = iter_p + 1 + i; j > 1 + i; --j)
	  fm_system_swap_column (s, j, j - 1);
      // Restore the ls_space_t structure with final sets.
      fm_solution_free (space->u_polyhedron[k]);
      fm_compsol_free (space->u_compacted[k]);
      s_fm_solution_t* tsol = fm_system_to_solution (s);
      space->u_polyhedron[k] = tsol;
      space->u_compacted[k] = fm_compsol_init_sol (tsol);
      fm_system_free (s);
    }

  // Optimize the space for exploration.
  if (options->normalize_space)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "... Optimize search space [1/2]\n");
      ls_schedspace_optimize_space (program, options, space);
    }

  for (k = 0; k < space->dimension; ++k)
    {
      fm_solution_cut (space->u_polyhedron[k], nb_param);
      fm_compsol_free (space->u_compacted[k]);
      space->u_compacted[k] = fm_compsol_init_sol (space->u_polyhedron[k]);
    }

  // Optimize the space for exploration.
  if (options->normalize_space)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "... Optimize search space [2/2]\n");
      ls_schedspace_optimize_space (program, options, space);
/*       ls_fusionspace_normalize_space (program, options, space); */
    }
  ls_fusionspace_normalize_space (program, options, space, 0, program->nb_statements - 1);

  // Internal conflict/dependence graph is no longer needed.
  ls_graph_free (graph);

  // Copy space for compatibility with former heuritics.
  space->polyhedron = space->u_polyhedron[0];

  return space;
}
