/*
 * graph.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/graph.h>


/**
 * Allocate a fresh (empty) graph.
 *
 *
 */
s_graph_t*
ls_graph_alloc ()
{
  s_graph_t* ret = XMALLOC(s_graph_t, 1);
  ret->root = NULL;
  ret->first_edge = NULL;
  ret->vertex_count = 0;
  ret->edge_count = 0;
  ret->tag = LS_GRAPH_TAG_UNDEFINED;
  ret->usr = NULL;
}


/**
 * Clear a graph.
 *
 *
 */
void
ls_graph_empty (s_graph_t* g)
{
  s_vertex_t* v;
  s_vertex_t* vnext;
  s_edge_t* e;
  s_edge_t* enext;

  if (g == NULL)
    return;

  // Free vertice.
  for (v = g->root; v; )
    {
      fm_list_free (v->in, fm_list_dummy_free);
      fm_list_free (v->out, fm_list_dummy_free);
      vnext = v->next;
      XFREE(v);
      v = vnext;
    }

  // Free edges.
  for (e = g->first_edge; e; )
    {
      enext = e->next;
      XFREE(e);
      e = enext;
    }

  g->edge_count = 0;
  g->vertex_count = 0;
  g->root = NULL;
  g->first_edge = NULL;
}


/**
 * Delete a graph.
 *
 *
 */
void
ls_graph_free (s_graph_t* g)
{
  // Empty graph.
  ls_graph_empty (g);
  // Free graph structure.
  XFREE(g);
}



/**
 * Dummy visitor for debug.
 *
 */
static
void
visit_printer (s_graph_t* g, s_vertex_t* v)
{
  printf ("S%d tag: %d\n", v->id, v->tag);
}


/**
 * Sample recursive exploration of the graph. walk bit of each
 * vertex to be traversed must be set to 0.
 *
 *
 */
void ls_graph_dfs_from (s_graph_t* g,
			s_vertex_t* v,
			visit_fun_t prefix,
			visit_fun_t infix,
			visit_fun_t suffix)
{
  if (v == NULL)
    return;

  v->walk = 1;

  // Prefix here.
  if (prefix)
    prefix (g, v);

  s_fm_list_t* l;
  s_edge_t* e;
  for (l = v->out; l; l = l->next)
    {
      e = l->data;
      if (e->dst->walk == 0)
	ls_graph_dfs_from (g, e->dst, prefix, infix, suffix);
    }

  // Suffix here.
  if (suffix)
    suffix (g, v);
}


/**
 * Graph traversal (DFS, recursive).
 *
 *
 */
void
ls_graph_dfs (s_graph_t* g,
	      visit_fun_t prefix,
	      visit_fun_t infix,
	      visit_fun_t suffix)
{
  s_vertex_t* v;

  // Reset the walk.
  for (v = g->root; v; v = v->next)
    v->walk = 0;

  // Recursive exploration from the root vertex. Iterate to all
  // unwalked vertice.
  for (v = g->root; v; v = v->next)
    if (v->walk == 0)
      ls_graph_dfs_from (g, v, prefix, infix, suffix);
}



/**
 * Sample recursive exploration of the graph. walk bit of each
 * vertex to be traversed must be set to 0.
 * Treats directed graphs as undirected.
 *
 */
void ls_graph_undirected_dfs_from (s_graph_t* g,
				   s_vertex_t* v,
				   visit_fun_t prefix,
				   visit_fun_t infix,
				   visit_fun_t suffix)
{
  if (v == NULL)
    return;

  v->walk = 1;

  // Prefix here.
  if (prefix)
    prefix (g, v);

  s_fm_list_t* l;
  s_edge_t* e;
  for (l = v->out; l; l = l->next)
    {
      e = l->data;
      if (e->dst->walk == 0)
	ls_graph_undirected_dfs_from (g, e->dst, prefix, infix, suffix);
    }
  for (l = v->in; l; l = l->next)
    {
      e = l->data;
      if (e->dst->walk == 0)
	ls_graph_undirected_dfs_from (g, e->dst, prefix, infix, suffix);
    }

  // Suffix here.
  if (suffix)
    suffix (g, v);
}


/**
 * Graph traversal (DFS, recursive).
 * Treats directed graphs as undirected.
 *
 */
void
ls_graph_undirected_dfs (s_graph_t* g,
			 visit_fun_t prefix,
			 visit_fun_t infix,
			 visit_fun_t suffix)
{
  s_vertex_t* v;

  // Reset the walk.
  for (v = g->root; v; v = v->next)
    v->walk = 0;

  // Recursive exploration from the root vertex. Iterate to all
  // unwalked vertice.
  for (v = g->root; v; v = v->next)
    if (v->walk == 0)
      ls_graph_dfs_from (g, v, prefix, infix, suffix);
}


/**
 * Print the graph (in DOT format)
 *
 *
 */
void
ls_graph_print (FILE* out_file, s_graph_t* g)
{
  int e_count;
  int v_count;
  s_vertex_t* tmpv;
  s_edge_t* tmpe;

  // Initialize DOT output.
  fprintf (out_file, "digraph G {\n");
  fprintf (out_file, "# Graph output\n");
  fprintf (out_file, "# Generated by LetSee\n");

  // Output all vertices.
  for (tmpv = g->root, v_count = 0; tmpv; tmpv = tmpv->next, ++v_count)
    fprintf (out_file, "D%d [label=\"D%d:c=%d,s=%d\"];\n",
	     tmpv->id, tmpv->id, tmpv->color, tmpv->scc);

  // Output all edges.
  for (tmpe = g->first_edge, e_count = 0; tmpe; tmpe = tmpe->next, ++e_count)
    fprintf (out_file, "D%d -> D%d [];\n",
	     tmpe->src->id, tmpe->dst->id);

  // Finalize the output.
  fprintf (out_file, "# Number of vertices = %d;\n", v_count);
  fprintf (out_file, "# Number of edges = %d;\n", e_count);
  fprintf (out_file, "}\n");
}


/**
 * Remove an existing vertex in the graph.
 *
 *
 */
int
ls_graph_remove_vertex (s_graph_t* g, s_vertex_t* v)
{
  if (g == NULL)
    assert (! "g is null");

  if (g->root == NULL)
    return 1;

  s_vertex_t* tmp;
  s_vertex_t* pred = NULL;
  s_edge_t* edg;
  s_fm_list_t* tmpe;
  s_fm_list_t* l;
  int k;

  // Find the node.
  if (g->root == v)
    g->root = v->next;
  else
    {
      for (tmp = g->root; tmp->next && tmp->next != v; tmp = tmp->next)
	pred = tmp;
      assert (tmp->next);
      tmp->next = v->next;
    }

  // Remove its in/out going edges.
  for (l = v->in, k = 0; k < 2; l = v->out, ++k)
    for (tmpe = l; tmpe; tmpe = tmpe->next)
      {
	edg = tmpe->data;
	ls_graph_remove_edge (g, edg);
      }
  fm_list_free (v->in, fm_list_dummy_free);
  fm_list_free (v->out, fm_list_dummy_free);

/*   // Remove all remaining edges that connect to the node. */
/*   printf ("here\n"); */
/*   for (edg = g->first_edge; edg; ) */
/*     { */
/*       s_edge_t* next = edg->next; */
/*       if (edg->src == v || edg->dst == v) */
/* 	ls_graph_remove_edge (g, edg); */
/*       edg = next; */
/*     } */
/*   printf ("There\n"); */

  XFREE(v);

  g->vertex_count--;

  return 0;
}


/**
 * Remove an existing edge in the graph.
 *
 *
 */
int
ls_graph_remove_edge (s_graph_t* g, s_edge_t* e)
{
  if (g == NULL)
    assert (! "g is null");

  if (g->root == NULL)
    return 1;

  e->src->d_out--;
  e->dst->d_in--;

  // Update source and destination lists.
  fm_list_remove (&(e->src->out), e);
  fm_list_remove (&(e->dst->in), e);

  // Update the edge list.
  s_edge_t* tmp;
  if (g->first_edge == e)
    g->first_edge = e->next;
  else
    {
      for (tmp = g->first_edge; tmp && tmp->next != e; tmp = tmp->next)
	;
      tmp->next = e->next;
    }

  XFREE (e);
  (g->edge_count)--;
}


/**
 * Create a new edge in the graph.
 *
 *
 */
s_edge_t*
ls_graph_create_edge (s_graph_t* g,
		      s_vertex_t* src,
		      s_vertex_t* dst,
		      void* data)
{
  s_edge_t* tmp;
  // Create the edge.
  s_edge_t* ret = XMALLOC(s_edge_t, 1);
  ret->src = src;
  ret->dst = dst;
  ret->label = -1;
  ret->tag = -1;
  ret->data = data;
  ret->next = NULL;

  // Update src and dst lists.
  fm_list_add_head (&(src->out), ret);
  fm_list_add_head (&(dst->in), ret);
  (src->d_out)++;
  (dst->d_in)++;

  g->edge_count++;
  // Update the edge list (tail insert).
/*   for (tmp = g->first_edge; tmp && tmp->next; tmp = tmp->next) */
/*     ; */
/*   if (! tmp) */
/*     g->first_edge = ret; */
/*   else */
/*     tmp->next = ret; */
  // Update the edge list (head insert).
  ret->next = g->first_edge;
  g->first_edge = ret;

  return ret;
}


/**
 * Create a vertex 'id' in a graph. If vertex 'id' already exists,
 * return it.
 *
 */
s_vertex_t*
ls_graph_create_vertex (s_graph_t* g, void* data, int id)
{
  s_vertex_t* tmp;
  s_vertex_t* pred;

  // Check if vertex id exists.
  for (tmp = g->root; tmp && tmp->id != id; tmp = tmp->next)
    pred = tmp;
  // Existing node. Return it.
  if (tmp != NULL)
    return tmp;
  // New node. Create it.
  s_vertex_t* ret = XMALLOC(s_vertex_t, 1);
  ret->in = NULL;
  ret->out = NULL;
  ret->label = -1;
  ret->data = data;
  ret->color = -1;
  ret->tag = -1;
  ret->scc = -1;
  ret->d_in = 0;
  ret->d_out = 0;
  ret->id = id;
  ret->next = NULL;
  g->vertex_count++;
  // Update the vertex list (tail insert).
  if (g->root == NULL)
    g->root = ret;
  else
    pred->next = ret;

  return ret;
}







/**
 * Graph coloring algorithm. Output the number of colors.
 *
 */
int
ls_graph_color_opt (s_graph_t* g, int mask)
{
  s_vertex_t* v;
  s_fm_list_t* l;
  int mark[g->vertex_count];
  int i;
  s_edge_t* e;
  int max = -1;
  int max_col;
  s_vertex_t* order[g->vertex_count];
  int j;

  // Reset colors, if LS_GRAPH_COLOR_RESET is set.
  if ((mask & LS_GRAPH_COLOR_RESET) != 0)
    for (v = g->root; v; v = v->next)
      v->color = -1;


  // Compute the traversal order.
  int max_conn = 0;
  int cur_conn;
  // Retrive maximum connexity.
  for (v = g->root; v; v = v->next)
    {
      cur_conn = v->d_in + v->d_out;
      max_conn = cur_conn > max_conn ? cur_conn : max_conn;
    }

  // Order w.r.t maximum connexity.
  j = 0;
  do
    for (v = g->root; v; v = v->next)
      if (v->d_in + v->d_out == max_conn)
	order[j++] = v;
  while (--max_conn >= 0);

  // Sort w.r.t d_in
  for (i = 0; i < g->vertex_count; ++i)
    for (j = i + 1; j < g->vertex_count; ++j)
      {
	if (order[i]->d_in + order[i]->d_out ==
	    order[j]->d_in + order[j]->d_out)
	  if (order[i]->d_in + order[i]->d_out >= 2)
	    if (order[i]->d_in < order[j]->d_in)
	      {
		v = order[i];
		order[i] = order[j];
		order[j] = v;
	      }
      }

  // Color the graph.
/*   for (v = g->root; v; v = v->next) */
  for (j = 0; j < g->vertex_count; ++j)
    {
      v = order[j];
      // If it is a single node, skip coloring.
      if (v->color != -1 || (v->in == NULL && v->out == NULL))
	continue;

      // Reset mark.
      for (i = 0; i < g->vertex_count; ++i)
	mark[i] = 0;
      // Compute the surrounding colors.
      for (l = v->in; l; l = l->next)
	{
	  e = l->data;
	  if (e->src->color != -1)
	    mark[e->src->color] = 1;
	}
      for (l = v->out; l; l = l->next)
	{
	  e = l->data;
	  if (e->dst->color != -1)
	    mark[e->dst->color] = 1;
	}
      // Find the first available color.
      for (i = 0; i < g->vertex_count && mark[i] != 0; ++i)
	;
      v->color = i;
      if (i > max)
	max = i;
    }

  // Extract the most represented color.
  max_col = ls_graph_max_color (g);
  // Use it to color independent nodes, or graph->tag, if defined.
  max_col = g->tag == LS_GRAPH_TAG_UNDEFINED ? max_col : g->tag;
  for (v = g->root; v; v = v->next)
    if (v->color == -1 && v->in == NULL && v->out == NULL)
      v->color = max_col;

  return max + 1;
}





/**
 * Graph coloring algorithm. Output the number of colors.
 *
 */
int
ls_graph_color (s_graph_t* g, int mask)
{
  return ls_graph_color_opt (g, mask);


  s_vertex_t* v;
  s_fm_list_t* l;
  int mark[g->vertex_count];
  int i;
  s_edge_t* e;
  int max = -1;
  int max_col;

  // Reset colors, if LS_GRAPH_COLOR_RESET is set.
  if ((mask & LS_GRAPH_COLOR_RESET) != 0)
    for (v = g->root; v; v = v->next)
      v->color = -1;

  // Color the graph.
  for (v = g->root; v; v = v->next)
    {
      // If it is a single node, skip coloring.
      if (v->color != -1 || (v->in == NULL && v->out == NULL))
	continue;

      // Reset mark.
      for (i = 0; i < g->vertex_count; ++i)
	mark[i] = 0;
      // Compute the surrounding colors.
      for (l = v->in; l; l = l->next)
	{
	  e = l->data;
	  if (e->src->color != -1)
	    mark[e->src->color] = 1;
	}
      for (l = v->out; l; l = l->next)
	{
	  e = l->data;
	  if (e->dst->color != -1)
	    mark[e->dst->color] = 1;
	}
      // Find the first available color.
      for (i = 0; i < g->vertex_count && mark[i] != 0; ++i)
	;
      v->color = i;
      if (i > max)
	max = i;
    }

  // Extract the most represented color.
  max_col = ls_graph_max_color (g);
  // Use it to color independent nodes.
  for (v = g->root; v; v = v->next)
    if (v->color == -1 && v->in == NULL && v->out == NULL)
      v->color = max_col;

  return max + 1;
}



/**
 * Extract the most represented color.
 *
 *
 */
int
ls_graph_max_color (s_graph_t* g)
{
  s_vertex_t* v;
  int colors[g->vertex_count];
  int i;
  int max = 0;
  int max_id = 0;

  for (i = 0; i < g->vertex_count; ++i)
    colors[i] = 0;

  for (v = g->root; v; v = v->next)
    if (v->color != -1)
      colors[v->color]++;

  for (i = 0; i < g->vertex_count; ++i)
    if (max < colors[i])
      {
	max = colors[i];
	max_id = i;
      }

  return max_id;
}



/**
 * \brief Return the second most represented color in a colored graph.
 *
 */
int
ls_graph_second_max_color (s_graph_t* g)
{
  s_vertex_t* v;
  int colors[g->vertex_count];
  int i;
  int max = 0;
  int max_id = 0;
  int prev_max_id = 0;

  for (i = 0; i < g->vertex_count; ++i)
    colors[i] = 0;

  for (v = g->root; v; v = v->next)
    if (v->color != -1)
      colors[v->color]++;

  for (i = 0; i < g->vertex_count; ++i)
    if (max < colors[i])
      {
	max = colors[i];
	prev_max_id = max_id;
	max_id = i;
      }

  return prev_max_id;
}



/**
 * Trim a graph.
 *
 *
 */
void
ls_graph_trim (s_graph_t* g)
{
  s_vertex_t* v;
  for_all_nodes(v, g)
    {
      if (v->d_in == 0 && v->d_out == 0)
	ls_graph_remove_vertex (g, v);
    }
}

/**
 * Transpose a graph.
 *
 *
 */
void
ls_graph_transpose (s_graph_t* g)
{
  s_vertex_t* v;
  s_edge_t* e;
  s_fm_list_t* tmp;

  // Revert the in and out lists for all vertice.
  for (v = g->root; v; v = v->next)
    {
      tmp = v->in;
      v->in = v->out;
      v->out = tmp;
    }

  // Revert src and dst for all edges.
  for (e = g->first_edge; e; e = e->next)
    {
      v = e->src;
      e->src = e->dst;
      e->dst = v;
    }
}


/**
 * Duplicate a graph.
 *
 *
 */
s_graph_t*
ls_graph_dup (s_graph_t* g)
{
  s_vertex_t* v;
  s_vertex_t* src;
  s_vertex_t* dst;
  s_edge_t* e;
  s_edge_t* edg;
  s_graph_t* ret = ls_graph_alloc ();

  // Create vertice.
  for (v = g->root; v; v = v->next)
    {
      src = ls_graph_create_vertex (ret, v->data, v->id);
      src->walk = v->walk;
      src->label = v->label;
      src->color = v->color;
      src->tag = v->tag;
      src->scc = v->scc;
    }

  // Create edges.
  for (e = g->first_edge; e; e = e->next)
    {
      src = ls_graph_create_vertex (ret, NULL, e->src->id);
      dst = ls_graph_create_vertex (ret, NULL, e->dst->id);
      edg = ls_graph_create_edge (ret, src, dst, e->data);
      edg->label = e->label;
      edg->tag = e->tag;
    }

  ret->tag = g->tag;

  return ret;
}


/**
 * Visitor (suffix) for the first SCC's algorithm DFS.
 *
 */
static
void
visit_suffix_scc(s_graph_t* g, s_vertex_t* v)
{
  v->tag = g->tag;
  g->tag++;
}


/**
 * Visitor (prefix) for the second SCC's algorithm DFS.
 *
 */
static
void
visit_number_scc(s_graph_t* g, s_vertex_t* v)
{
  v->scc = g->tag;
}



/**
 * Modyfing ordering of vertice in a graph, topologically.
 *
 */
void
ls_graph_topological_apply (s_graph_t* g)
{
  s_vertex_t* tab[g->vertex_count];
  s_vertex_t* v;
  int i;

  if (g->root == NULL)
    return;

  if (! ls_graph_topological_sort (g))
    {
      for (v = g->root; v; v = v->next)
	tab[v->tag] = v;

      g->root = tab[0];
      for (i = 0; i < g->vertex_count - 1; ++i)
	tab[i]->next = tab[i + 1];
      tab[i]->next = NULL;
/*       g->root = tab[g->vertex_count - 1]; */
/*       for (i = 0; i < g->vertex_count - 1; ++i) */
/* 	tab[g->vertex_count - 1 - i]->next = tab[g->vertex_count - 1 - i - 1]; */
/*       tab[g->vertex_count - 1 - i]->next = NULL; */
    }
}


/**
 * Modyfing ordering of vertice in a graph, topologically.
 *
 */
void
ls_graph_topological_apply_depth (s_graph_t* g)
{
  s_vertex_t* tab[g->vertex_count];
  s_vertex_t* v;
  s_vertex_t* tmp;
  int i, j;
  CandlDependence* d1;
  CandlDependence* d2;

  if (g->root == NULL)
    return;

  if (! ls_graph_topological_sort (g))
    {
      for (v = g->root; v; v = v->next)
	tab[v->tag] = v;

      // Bubble sort on the depth of dependences.
      for (i = 0; i < g->vertex_count; ++i)
	for (j = i + 1; j < g->vertex_count; ++j)
	  {
	    d1 = tab[i]->data;
	    d2 = tab[j]->data;
	    if (d1->depth < d2->depth)
	      {
		tmp = tab[i];
		tab[i] = tab[j];
		tab[j] = tmp;
	      }
	  }

      g->root = tab[0];
      for (i = 0; i < g->vertex_count - 1; ++i)
	tab[i]->next = tab[i + 1];
      tab[i]->next = NULL;
/*       g->root = tab[g->vertex_count - 1]; */
/*       for (i = 0; i < g->vertex_count - 1; ++i) */
/* 	tab[g->vertex_count - 1 - i]->next = tab[g->vertex_count - 1 - i - 1]; */
/*       tab[g->vertex_count - 1 - i]->next = NULL; */
    }
}


/**
 * Topological ordering of vertice in a graph. Ordering is stored in
 * the vertex tag.
 * Return 0 if all nodes where visited, 1 elsewhere (meaning there was
 * a cycle).
 *
 */
int
ls_graph_topological_sort (s_graph_t* g)
{
  int i;
  int order = 0;
  s_vertex_t* v;
  s_edge_t* e;
  s_fm_list_t* tmp;
  s_fm_list_t* tmp2;

  // Reset the edge tag flag.
  for (e = g->first_edge; e; e = e->next)
    e->tag = -1;

  // Allocate the pseudo-stack (pessimistic).
  s_vertex_t* stack[g->vertex_count];
  int stack_top = 0;
  for (i = 0; i < g->vertex_count; ++i)
    stack[i] = NULL;

  // Push all nodes with no imcoming edges.
  for (v = g->root; v; v = v->next)
    if (v->in == NULL)
      stack[stack_top++] = v;

  // Topological sort.
  while (stack_top)
    {
      v = stack[--stack_top];
      v->tag = order++;
      for (tmp = v->out; tmp; tmp = tmp->next)
	{
	  e = tmp->data;
	  // 'Mark' the edge as deleted.
	  e->tag = 0;
	  // Check for remaining incoming edges in dst.
	  for (tmp2 = e->dst->in;
	       tmp2 && ((s_edge_t*)(tmp2->data))->tag == 0; tmp2 = tmp2->next)
	    ;
	  // No incoming edge remain, add dst to the stack.
	  if (! tmp2)
	    stack[stack_top++] = e->dst;
	}
    }

  return (g->vertex_count != order);
}


/**
 * Build a compacted graph, according to the previous SCC pass.
 *
 */
s_graph_t*
ls_graph_scc_build_dag (s_graph_t* g)
{
  s_graph_t* c = ls_graph_dup (g);
  int i, k;
  s_vertex_t* scc[c->tag];
  s_vertex_t* v;
  s_vertex_t* next;
  s_fm_list_t* tmp;
  s_fm_list_t* l;
  s_edge_t* e;

  // Build a compacted graph.
  for (i = 0; i < c->tag; ++i)
    scc[i] = NULL;
  for (v = c->root; v; )
    {
      next = v->next;
      if (scc[v->scc] == NULL)
	{
	  scc[v->scc] = v;
	  // Remove cycles.
	  for (l = v->in, k = 0; k < 2; l = v->out, ++k)
	    for (tmp = l; tmp; tmp = tmp->next)
	      {
		e = tmp->data;
		if (e->src->scc == e->dst->scc)
		  ls_graph_remove_edge (c, e);
	      }
	}
      else
	{
	  // Merge the node.
	  for (l = v->in, k = 0; k < 2; l = v->out, ++k)
	    for (tmp = l; tmp; tmp = tmp->next)
	      {
		e = tmp->data;
		if (e->src->scc != e->dst->scc)
		  {
		    if (l == v->in)
		      ls_graph_create_edge (c, e->src, scc[v->scc], e->data);
		    else
		      ls_graph_create_edge (c, scc[v->scc], e->dst, e->data);
		  }
	      }
	  ls_graph_remove_vertex (c, v);
	}
      v = next;
    }

  return c;
}


/**
 * Arrange SCC indices to be compliant with SCC topological ordering.
 * g->tag is supposed to contain the number of SCCs.
 *
 */
static
void
ls_graph_scc_topological (s_graph_t* g)
{
  // Build DAG from SCC representation.
  s_graph_t* c = ls_graph_scc_build_dag (g);

  int scc_map[c->tag];
  s_vertex_t* v;

  // Apply topological order (this is now a DAG).
  ls_graph_topological_sort (c);

  // Update the SCC indices.
  for (v = c->root; v; v = v->next)
    scc_map[v->scc] = v->tag;
  for (v = g->root; v; v = v->next)
    v->scc = scc_map[v->scc];

  // Be clean.
  ls_graph_free (c);
}


/**
 * Compute SCC for a given graph. The SCC id of each vertex is
 * stored in the scc flag of the vertex, with topological ordering.
 * Output the number of SCC in the graph.
 *
 */
int
ls_graph_compute_scc (s_graph_t* g)
{
  s_vertex_t* v;
  s_vertex_t* maxv;
  int max;

  // First DFS: number the vertices in order of the completion of
  // the recursive call (suffix)
  g->tag = 0;
  ls_graph_dfs (g, NULL, NULL, visit_suffix_scc);

  // Transpose the graph.
  ls_graph_transpose (g);

  // Second DFS: starts from the highest vertex tag. If unreached vertex
  // remain, start again.
  // Reinitialize the walk bit.
  for (v = g->root; v; v = v->next)
    v->walk = 0;
  g->tag = 0;
  do
    {
      max = -1;
      // Compute the next root.
      for (v = g->root; v; v = v->next)
	if (v->walk == 0 && max < v->tag)
	  {
	    max = v->tag;
	    maxv = v;
	  }
      // If unvisited vertex remains, traverse the graph.
      if (max != -1)
	{
	  ls_graph_dfs_from (g, maxv, visit_number_scc, NULL, NULL);
	  g->tag++;
	}
    }
  while (max != -1);

  // Transpose back the graph.
  ls_graph_transpose (g);

  // Order SCC indices topologically.
  ls_graph_scc_topological (g);

  return g->tag;
}



/**
 * DEBUG: print SCCs.
 *
 */
static
void
visit_prefix_printvertex (s_graph_t* g, s_vertex_t* v)
{
  if (v->scc == g->tag)
    printf ("S%d ", v->id);
}


void
ls_graph_print_scc (s_graph_t* g, int nb_scc)
{
  int i;

  for (i = 0; i < nb_scc; ++i)
    {
      g->tag = i;
      printf ("%d: ", i);
      ls_graph_dfs (g, visit_prefix_printvertex, NULL, NULL);
      printf ("\n");
    }
}
