/*
 * heuristic-h1.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif


#include <letsee/heuristic-h1.h>


static int GCOUNT;


/**
 * Basic function to relax the bound of a variable traversed (used
 * when facing integer holes).
 *
 */
static
int
relax1 (s_fm_solution_t* sol,
	s_fm_vector_t* draw,
	z_type_t* min,
	z_type_t* max,
	int* idx,
	int* toggle)
{
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;

  fm_solver_compute_min (&lb, sol->solution[*idx].positive,
			 draw, *idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[*idx].negative,
			 draw, *idx, FM_MINLEXICO_INT);

  if (*toggle)
    {
      if (Z_CMP(lb->num, <, min[*idx]))
	Z_DEC(min[*idx], min[*idx]);
      else if (Z_CMP(Ub->num, >, max[*idx]))
	Z_INC(max[*idx], max[*idx]);
      else
	return LS_EXIT_ERROR;
    }
  else
    {
      if (Z_CMP(Ub->num, >, max[*idx]))
	Z_INC(max[*idx], max[*idx]);
      else if (Z_CMP(lb->num, <, min[*idx]))
	Z_DEC(min[*idx], min[*idx]);
      else
	return LS_EXIT_ERROR;
    }

  *toggle = ! *toggle;
  return LS_EXIT_OK;
}



/**
 * Schedule finder (implicit correction algorithm -- unpublished).
 *
 * Input:  - A prototype schedule (prototype)
 *         - A relaxation function (relax)
 * Output: The closest possible point (draw) in the solution polytope.
 *
 *
 */
static
s_fm_vector_t*
ls_explorer_schedfinder (s_ls_space_t* space,
			 CandlProgram* program,
			 s_ls_options_t* options,
			 s_fm_vector_t* prototype,
			 relax_function_t relax)
{
  s_fm_solution_t* sol = space->polyhedron;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  int idx = 0;
  s_fm_vector_t* draw = fm_vector_alloc (space->polyhedron->size + 1);
  int toggle = 0;

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;

  // Ensure there is no conflict on the first dimension,
  if (Z_CMP(min[idx], >, max[idx]))
    return NULL;

  // Assign the closest bound to the prototype for the schedule.
  if (Z_CMP(prototype->vector[idx + 1].num, <, min[idx]))
    Z_ASSIGN(max[idx], min[idx]);
  else if (Z_CMP(prototype->vector[idx + 1].num, <=, max[idx]))
    {
	Z_ASSIGN(min[idx], prototype->vector[idx + 1].num);
	Z_ASSIGN(max[idx], prototype->vector[idx + 1].num);
    }
  else
    Z_ASSIGN(min[idx], max[idx]);

  // Iterate on all dimensions.
  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
      // The solution is reached.
      if (idx + 1 == sol->size)
	{
/* 	  ls_transformations_generate (draw, program, options, */
/* 				       space->size); */
/* 	  // FIXME: Here for debugging purposes. */
/* 	  fm_vector_print (stdout, prototype); printf ("\n"); */
/* 	  fm_vector_print (stdout, draw); printf ("\n"); */
	}
      else
	{
	  // Go to the next dimension.
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);

	  // A contradiction occurs.
	  if (Z_CMP(lb->num, >, Ub->num))
	    {
	      // Go to the previous dimension.
	      --idx;
	      // We can not iterate with the current bounds, so we
	      // have to change the bounds or to find the first
	      // correct dimension for iteration.
	      while (idx >= 0 && Z_CMP(i[idx], ==, max[idx]))
		{
		  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
					 draw, idx, FM_MINLEXICO_INT);
		  // We reached the real upper bound for the
		  // dimension, go to the previous one.
		  if (Z_CMP(Ub->num, ==, max[idx]))
		    --idx;
		  else
		    {
		      // Relax the current dimension bounds.
		      relax (sol, draw, min, max, &idx, &toggle);
		      Z_ASSIGN(i[idx], min[idx]);
		      Z_DEC(i[idx], i[idx]);
		    }
		}
	      if (idx < 0)
		{
		  idx = 0;
		  Z_DEC(max[idx], max[idx]);
		  Z_ASSIGN(i[idx], max[idx]);
		  Z_DEC(i[idx], i[idx]);
/* 		  fprintf (stderr, "Error in correction.\n"); */
/* 		  exit (1); */
		}
	    }
	  else
	    {
	      // No contradiction occurs, we try to be as close as
	      // possible to the prototype schedule.
	      if (Z_CMP(prototype->vector[idx + 1].num, <, lb->num))
		Z_ASSIGN(max[idx], lb->num);
	      else if (Z_CMP(prototype->vector[idx + 1].num, <=, Ub->num))
		Z_ASSIGN(max[idx], prototype->vector[idx + 1].num);
	      else Z_ASSIGN(max[idx], Ub->num);

	      Z_ASSIGN(i[idx], max[idx]);
	      Z_DEC(i[idx], i[idx]);
	    }
	}
    }

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);

  return draw;
}


/**
 *
 * Wrapper for ls_explorer_schedfinder for multidimensional schedules.
 *
 *
 */
static
s_fm_vector_t*
ls_explorer_schedfinder_multi (s_ls_space_t* space,
			       CandlProgram* program,
			       s_ls_options_t* options,
			       s_fm_vector_t* prototype,
			       relax_function_t relax,
			       int dim)
{
  space->polyhedron = space->u_polyhedron[dim];

  return ls_explorer_schedfinder (space, program, options, prototype, relax);
}



/**
 * H1 heuristic (officially Decoupling Heuristic). Part related to
 * parameter traversal.
 *
 */
static
void
ls_explorer_h1m_constant (s_ls_space_t* space,
			   CandlProgram* program,
			   s_ls_options_t* options,
			   s_ls_explorer_t* data)
{
  int tested = 0;
  int nb_param = 0;
  int param_count;
  int* param_idx;
  // j is the current schedule dimension.
  int i, j, k;
  int value = 0;
  int count = 0;
  s_fm_vector_t** prototype = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  int* idx = XMALLOC(int, space->dimension);
  unsigned cache_hit = 0;
  z_type_t** ii = XMALLOC(z_type_t*, space->dimension);
  z_type_t** min = XMALLOC(z_type_t*, space->dimension);
  z_type_t** max = XMALLOC(z_type_t*, space->dimension);
  s_fm_vector_t** transfo = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  transfo[space->dimension] = NULL;
  prototype[space->dimension] = NULL;
  int hardlimit = options->rtries;
  s_fm_rational_t* rat = fm_rational_alloc ();
  int go;

  // Count the number of parameters.
  param_count = program->statement[0]->domain->NbColumns -
    program->statement[0]->depth - 2;
  nb_param = program->nb_statements;

  // Initialize the explorer iterators.
  for (j = 0; j < space->dimension; ++j)
    {
      idx[j] = 0;
      ii[j] = XMALLOC(z_type_t, nb_param);
      min[j] = XMALLOC(z_type_t, nb_param);
      max[j] = XMALLOC(z_type_t, nb_param);
      for (i = 0; i < nb_param; ++i)
	{
	  Z_INIT(ii[j][i]);
	  Z_INIT(min[j][i]);
	  Z_INIT(max[j][i]);
	}
    }

  // Create and assign the array of parameters indexes.
  param_idx = XMALLOC(int, nb_param);

  for (value = 0, i = 0; i < program->nb_statements; ++i)
    value += program->statement[i]->depth;
  for (i = 0; i < program->nb_statements; ++i)
    value += param_count;
  for (i = 0; i < program->nb_statements; ++i)
    //    for (j = 0; j < param_count; ++j)
      param_idx[count++] = value++;

  // Generate the schedules within the min/max bounds for the paramator parts.
  count = data->count;
  for (i = 0; i < count; ++i)
    {
      for (j = 0; j < space->dimension; ++j)
	{
	  prototype[j] = fm_vector_dup (data->u_draws[i][j]);
	  fm_solver_compute_min
	    (&rat,
	     space->u_polyhedron[j]->solution[param_idx[0]].positive,
	     prototype[j], param_idx[0], FM_MINLEXICO_INT);
	  min[j][0] = rat->num;
	  fm_solver_compute_max
	    (&rat,
	     space->u_polyhedron[j]->solution[param_idx[0]].negative,
	     prototype[j], param_idx[0], FM_MINLEXICO_INT);
	  max[j][0] = rat->num;
	  idx[j] = 0;
	}
      j = 0; go = 1;
      for (Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
	   Z_CMP(ii[j][idx[j]], <=, max[j][idx[j]]) && go;
	   Z_INC(ii[j][idx[j]], ii[j][idx[j]]))
	{
	  fm_vector_assign_int_idx (prototype[j], ii[j][idx[j]],
				    param_idx[idx[j]] + 1);
	  if (idx[j] == nb_param - 1)
	    {
	      // Generate transformation.
	      transfo[j] =
		ls_explorer_schedfinder_multi (space, program, options,
					       prototype[j], relax1, j);
/* 	      ls_explorer_transfo_print (transfo); */
	      if (j == space->dimension - 1)
		{
		  // Generate the transformation.
		  // Ensure the transformation has not already been generated.
		  if (! ls_explorer_transfo_exists_multi (data, transfo))
		    {
		      if (hardlimit-- == 0)
			break;
		      // Memorize and execute it.
		      ls_explorer_transfo_copy (&(data->u_draws[data->count]),
						transfo);

		      data->cycles[data->count] =
			ls_explorer_exectrans (program, options, transfo,
					       data->count);
		      printf ("%d %Ld\n", ++GCOUNT, data->cycles[data->count]);
		      data->ids[data->count] = data->count;
/* 		      if (options->verbose) */
/* 			{ */
/* 			  fprintf (options->out_file, "*"); */
/* 			  fflush (options->out_file); */
/* 			} */
		      if (data->cycles[data->count] != 0)
			{
			  data->count++;
			  ++tested;
			}
		    }
		  else
		    {
/* 		      printf ("Transfo exists!\n"); */
		      ++cache_hit;
		    }

		  // Go back.
		  int mybool = 1;
		  while (mybool)
		    {
		      // Max value in legal bound is attained for
		      // variable indexed at schedule dimension j.
		      while (idx[j] >= 0 &&
			     Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]]))
			{
			  // Go to the previous index.
			  Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
			  --(idx[j]);
			}
		      if (idx[j] < 0)
			{
			  // Done traversing this dimension.
			  idx[j] = 0;
			  --j;
			  if (j < 0)
			    {
			      //break, traversal finished.
			      go = 0;
			      mybool = 0;
			      j = 0;
			      //printf ("BREAKING\n");
			    }
			}
		      else
			mybool = 0;
		    }
		  for (k = j; k < space->dimension; ++k)
		    fm_vector_free (transfo[k]);
		}
	      else
		{
		  // Go to the next dimension.
		  ++j;
		  idx[j] = 0;
		}
	    }
	  else
	    {
	      ++(idx[j]);
	      fm_solver_compute_min
		(&rat,
		 space->u_polyhedron[j]->solution[idx[j]].positive,
		 prototype[j], idx[j], FM_MINLEXICO_INT);
	      Z_ASSIGN(min[j][idx[j]], rat->num);
	      fm_solver_compute_max
		(&rat,
		 space->u_polyhedron[j]->solution[idx[j]].negative,
		 prototype[j], idx[j], FM_MINLEXICO_INT);
	      Z_ASSIGN(max[j][idx[j]], rat->num);
	      Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
	      Z_DEC(ii[j][idx[j]], ii[j][idx[j]]);
	      while (idx[j] > 0 && (Z_CMP(min[j][idx[j]], >, max[j][idx[j]]) ||
				    Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]])))
		{
		  --(idx[j]);
		}
	    }
	}
      for (j = 0; j < space->dimension; ++j)
	{
	  fm_vector_free (prototype[j]);
	  idx[j] = 0;
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "\n.... Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, ".... Number of cache hits: %d\n",
	       cache_hit);
    }

  // Be clean.
  for (j = 0; j < space->dimension; ++j)
    {
      for (i = 0; i < nb_param; ++i)
	{
	  Z_CLEAR(ii[j][i]);
	  Z_CLEAR(min[j][i]);
	  Z_CLEAR(max[j][i]);
	}
      XFREE(ii[j]);
      XFREE(min[j]);
      XFREE(max[j]);
    }
  XFREE(ii);
  XFREE(min);
  XFREE(max);
  fm_rational_free (rat);
}



/**
 * H1 heuristic (officially Decoupling Heuristic). Part related to
 * parameter traversal.
 *
 */
static
void
ls_explorer_h1m_parameter (s_ls_space_t* space,
			   CandlProgram* program,
			   s_ls_options_t* options,
			   s_ls_explorer_t* data)
{
  int tested = 0;
  int nb_param = 0;
  int param_count;
  int* param_idx;
  // j is the current schedule dimension.
  int i, j, k;
  int value = 0;
  int count = 0;
  s_fm_vector_t** prototype = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  int* idx = XMALLOC(int, space->dimension);
  unsigned cache_hit = 0;
  z_type_t** ii = XMALLOC(z_type_t*, space->dimension);
  z_type_t** min = XMALLOC(z_type_t*, space->dimension);
  z_type_t** max = XMALLOC(z_type_t*, space->dimension);
  s_fm_vector_t** transfo = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  transfo[space->dimension] = NULL;
  prototype[space->dimension] = NULL;
  int hardlimit = options->rtries;
  s_fm_rational_t* rat = fm_rational_alloc ();
  int go;

  // Count the number of parameters.
  param_count = program->statement[0]->domain->NbColumns -
    program->statement[0]->depth - 2;
  nb_param = param_count * program->nb_statements;

  // Initialize the explorer iterators.
  for (j = 0; j < space->dimension; ++j)
    {
      idx[j] = 0;
      ii[j] = XMALLOC(z_type_t, nb_param);
      min[j] = XMALLOC(z_type_t, nb_param);
      max[j] = XMALLOC(z_type_t, nb_param);
      for (i = 0; i < nb_param; ++i)
	{
	  Z_INIT(ii[j][i]);
	  Z_INIT(min[j][i]);
	  Z_INIT(max[j][i]);
	}
    }

  // Create and assign the array of parameters indexes.
  param_idx = XMALLOC(int, nb_param);

  for (value = 0, i = 0; i < program->nb_statements; ++i)
    value += program->statement[i]->depth;
  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < param_count; ++j)
      param_idx[count++] = value++;

  // Generate the schedules within the min/max bounds for the paramator parts.
  count = data->count;
  for (i = 0; i < count; ++i)
    {
      for (j = 0; j < space->dimension; ++j)
	{
	  prototype[j] = fm_vector_dup (data->u_draws[i][j]);
	  fm_solver_compute_min
	    (&rat,
	     space->u_polyhedron[j]->solution[param_idx[0]].positive,
	     prototype[j], param_idx[0], FM_MINLEXICO_INT);
	  min[j][0] = rat->num;
	  fm_solver_compute_max
	    (&rat,
	     space->u_polyhedron[j]->solution[param_idx[0]].negative,
	     prototype[j], param_idx[0], FM_MINLEXICO_INT);
	  max[j][0] = rat->num;
	  idx[j] = 0;
	}
      j = 0; go = 1;
      for (Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
	   Z_CMP(ii[j][idx[j]], <=, max[j][idx[j]]) && go;
	   Z_INC(ii[j][idx[j]], ii[j][idx[j]]))
	{
	  fm_vector_assign_int_idx (prototype[j], ii[j][idx[j]],
				    param_idx[idx[j]] + 1);
	  if (idx[j] == nb_param - 1)
	    {
	      // Generate transformation.
	      transfo[j] =
		ls_explorer_schedfinder_multi (space, program, options,
					       prototype[j], relax1, j);
/* 	      ls_explorer_transfo_print (transfo); */
	      if (j == space->dimension - 1)
		{
		  // Generate the transformation.
		  // Ensure the transformation has not already been generated.
		  if (! ls_explorer_transfo_exists_multi (data, transfo))
		    {
		      if (hardlimit-- == 0)
			break;
		      // Memorize and execute it.
		      ls_explorer_transfo_copy (&(data->u_draws[data->count]),
						transfo);

		      data->cycles[data->count] =
			ls_explorer_exectrans (program, options, transfo,
					       data->count);
		      printf ("%d %Ld\n", ++GCOUNT, data->cycles[data->count]);
		      data->ids[data->count] = data->count;
/* 		      if (options->verbose) */
/* 			{ */
/* 			  fprintf (options->out_file, "*"); */
/* 			  fflush (options->out_file); */
/* 			} */
		      if (data->cycles[data->count] != 0)
			{
			  data->count++;
			  ++tested;
			}
		    }
		  else
		    {
/* 		      printf ("Transfo exists!\n"); */
		      ++cache_hit;
		    }

		  // Go back.
		  int mybool = 1;
		  while (mybool)
		    {
		      // Max value in legal bound is attained for
		      // variable indexed at schedule dimension j.
		      while (idx[j] >= 0 &&
			     Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]]))
			{
			  // Go to the previous index.
			  Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
			  --(idx[j]);
			}
		      if (idx[j] < 0)
			{
			  // Done traversing this dimension.
			  idx[j] = 0;
			  --j;
			  if (j < 0)
			    {
			      //break, traversal finished.
			      go = 0;
			      mybool = 0;
			      j = 0;
			      //printf ("BREAKING\n");
			    }
			}
		      else
			mybool = 0;
		    }
		  for (k = j; k < space->dimension; ++k)
		    fm_vector_free (transfo[k]);
		}
	      else
		{
		  // Go to the next dimension.
		  ++j;
		  idx[j] = 0;
		}
	    }
	  else
	    {
	      ++(idx[j]);
	      fm_solver_compute_min
		(&rat,
		 space->u_polyhedron[j]->solution[idx[j]].positive,
		 prototype[j], idx[j], FM_MINLEXICO_INT);
	      Z_ASSIGN(min[j][idx[j]], rat->num);
	      fm_solver_compute_max
		(&rat,
		 space->u_polyhedron[j]->solution[idx[j]].negative,
		 prototype[j], idx[j], FM_MINLEXICO_INT);
	      Z_ASSIGN(max[j][idx[j]], rat->num);
	      Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
	      Z_DEC(ii[j][idx[j]], ii[j][idx[j]]);
	      while (idx[j] > 0 && (Z_CMP(min[j][idx[j]], >, max[j][idx[j]]) ||
				    Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]])))
		{
		  --(idx[j]);
		}
	    }
	}
      for (j = 0; j < space->dimension; ++j)
	{
	  fm_vector_free (prototype[j]);
	  idx[j] = 0;
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "\n.... Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, ".... Number of cache hits: %d\n",
	       cache_hit);
    }

  // Be clean.
  for (j = 0; j < space->dimension; ++j)
    {
      for (i = 0; i < nb_param; ++i)
	{
	  Z_CLEAR(ii[j][i]);
	  Z_CLEAR(min[j][i]);
	  Z_CLEAR(max[j][i]);
	}
      XFREE(ii[j]);
      XFREE(min[j]);
      XFREE(max[j]);
    }
  XFREE(ii);
  XFREE(min);
  XFREE(max);
  fm_rational_free (rat);
}


/**
 * FIXME: Debug it! Was roughly replaced by the parameters' version.
 *
 */
static
void
ls_explorer_h1m_constant_SEGV (s_ls_space_t* space,
			       CandlProgram* program,
			       s_ls_options_t* options,
			       s_ls_explorer_t* data)
{
  int tested = 0;
  int nb_const = 0;
  int* const_idx;
  int i, j;
  int value = 0;
  int count = 0;
  s_fm_vector_t** prototype = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  int* idx = XMALLOC(int, space->dimension);
  unsigned cache_hit = 0;
  z_type_t** ii = XMALLOC(z_type_t*, space->dimension);
  s_fm_vector_t** transfo = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  transfo[space->dimension] = NULL;
  prototype[space->dimension] = NULL;
  int hardlimit = options->rtries;

  // Count the number of constants.
  nb_const = program->nb_statements;

  for (j = 0; j < space->dimension; ++j)
    {
      idx[j] = 0;
      ii[j] = XMALLOC(z_type_t, nb_const);
      for (i = 0; i < nb_const; ++i)
	Z_INIT(ii[j][i]);
    }

  // Create and assign the array of constant indexes.
  const_idx = XMALLOC(int, nb_const);

  // Compute the starting index of constants.
  value = program->statement[0]->domain->NbColumns -
    program->statement[0]->depth - 2;
  value = value * program->nb_statements;
  for (i = 0; i < program->nb_statements; ++i)
    value += program->statement[i]->depth;

  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < nb_const; ++j)
      const_idx[count++] = value++;

  // Generate the schedules within the min/max bounds for the constant parts.
  count = data->count;
  for (i = 0; i < count; ++i)
    {
      for (j = 0; j < space->dimension; ++j)
	prototype[j] = fm_vector_dup (data->u_draws[i][j]);
      j = 0;
      for (Z_ASSIGN(ii[j][idx[j]], options->lb);
	   Z_CMP(ii[j][idx[j]], <=, options->Ub);
	   Z_INC(ii[j][idx[j]], ii[j][idx[j]]))
	{
	  fm_vector_assign_int_idx (prototype[j], ii[j][idx[j]],
				    const_idx[idx[j]] + 1);
	  if (idx[j] == nb_const - 1)
	    {
	      // Generate transformation.
	      transfo[j] =
		ls_explorer_schedfinder_multi (space, program, options,
					       prototype[j], relax1, j);
	      if (j == space->dimension - 1)
		{
		  // Generate the transformation.
		  // Ensure the transformation has not already been generated.
		  if (! ls_explorer_transfo_exists_multi (data, transfo))
		    {
		      if (hardlimit-- == 0)
			break;
		      // Memorize and execute it.
		      ls_explorer_transfo_copy (&(data->u_draws[data->count]),
						transfo);
		      data->cycles[data->count] =
			ls_explorer_exectrans (program, options, transfo,
					       data->count);
		      printf ("%d %Ld\n", ++GCOUNT, data->cycles[data->count]);
		      data->ids[data->count] = data->count;
/* 		      if (options->verbose) */
/* 			{ */
/* 			  fprintf (options->out_file, "*"); */
/* 			  fflush (options->out_file); */
/* 			} */
		      if (data->cycles[data->count] != 0)
			{
			  data->count++;
			  ++tested;
			}
		    }
		  else
		    {
		      for (i = 0; i < space->dimension; ++i)
			fm_vector_free (transfo[i]);
		      ++cache_hit;
		    }

		  // Go back.
		  while (j >= 0)
		    {
		      while (idx[j] >= 0 &&
			     Z_CMP(ii[j][idx[j]], ==, options->Ub))
			{
			  Z_ASSIGN(ii[j][idx[j]], options->lb);
			  --(idx[j]);
			}
		      if (idx[j] < 0)
			break;
		      --j;
		    }
		  j = 0;
		}
	      else
		{
		  // Go to the next dimension.
		  ++j;
		}
	    }
	  else
	    {
	      ++(idx[j]);
	      Z_ASSIGN(ii[j][idx[j]], options->lb);
	      Z_DEC(ii[j][idx[j]], ii[j][idx[j]]);
	    }
	}
      for (j = 0; j < space->dimension; ++j)
	{
	  fm_vector_free (prototype[j]);
	  idx[j] = 0;
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "\n.... Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, ".... Number of cache hits: %d\n",
	       cache_hit);
    }

  for (j = 0; j < space->dimension; ++j)
    {
      for (i = 0; i < nb_const; ++i)
	Z_CLEAR(ii[j][i]);
      XFREE(ii[j]);
    }
  XFREE(ii);
}




static
void
ls_explorer_h1m_iterator (s_ls_space_t* space,
			  CandlProgram* program,
			  s_ls_options_t* options,
			  s_fm_vector_t** prototype,
			  s_ls_explorer_t* data)
{
  int tested = 0;
  int nb_iter = 0;
  int* iter_idx;
  int i, j, k;
  int value = 0;
  int count = 0;
  int* idx = XMALLOC(int, space->dimension);
  unsigned cache_hit = 0;
  z_type_t** ii = XMALLOC(z_type_t*, space->dimension);
  z_type_t** min = XMALLOC(z_type_t*, space->dimension);
  z_type_t** max = XMALLOC(z_type_t*, space->dimension);
  s_fm_vector_t** transfo = XMALLOC(s_fm_vector_t*, space->dimension + 1);
  transfo[space->dimension] = NULL;
  int hardlimit = options->rtries;
  s_fm_rational_t* rat = fm_rational_alloc ();

  int j_mem;

  // Count the number of iterators.
  for (i = 0; i < program->nb_statements; ++i)
    nb_iter += program->statement[i]->depth;

  // Initialize the explorer iterators.
  for (j = 0; j < space->dimension; ++j)
    {
      idx[j] = 0;
      ii[j] = XMALLOC(z_type_t, nb_iter);
      min[j] = XMALLOC(z_type_t, nb_iter);
      max[j] = XMALLOC(z_type_t, nb_iter);
      for (i = 0; i < nb_iter; ++i)
	{
	  Z_INIT(ii[j][i]);
	  Z_INIT(min[j][i]);
	  Z_INIT(max[j][i]);
	}
      fm_solver_compute_min (&rat,
			     space->u_polyhedron[j]->solution[0].positive,
			     prototype[j], 0, FM_MINLEXICO_INT);
      min[j][0] = rat->num;
      fm_solver_compute_max (&rat,
			     space->u_polyhedron[j]->solution[0].negative,
			     prototype[j], 0, FM_MINLEXICO_INT);
      max[j][0] = rat->num;
    }


  // Create and assign the array of iterator indexes.
  iter_idx = XMALLOC(int, nb_iter);

  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < program->statement[i]->depth; ++j)
      {
	iter_idx[count++] = value;
	++value;
      }

  j = 0;
  // Generate the schedules within the min/max bounds for the iterator parts.
  for (Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
       Z_CMP(ii[j][idx[j]], <=, max[j][idx[j]]);
       Z_INC(ii[j][idx[j]], ii[j][idx[j]]))
    {
      fm_vector_assign_int_idx (prototype[j], ii[j][idx[j]],
				iter_idx[idx[j]] + 1);
      if (idx[j] == nb_iter - 1)
	{
	  // Generate transformation.
	    transfo[j] =
	      ls_explorer_schedfinder_multi (space, program, options,
					     prototype[j], relax1, j);
	  if (j == space->dimension - 1)
	    {
	      // Generate the transformation.
	      // Ensure the transformation has not already been generated.
	      if (! ls_explorer_transfo_exists_multi (data, transfo))
		{
		  if (hardlimit-- == 0)
		    break;

		  // Memorize and execute it.
		  ls_explorer_transfo_copy (&(data->u_draws[data->count]),
					    transfo);
		  data->cycles[data->count] =
		    ls_explorer_exectrans (program, options, transfo,
					   data->count);
		  printf ("%d %Ld\n", ++GCOUNT, data->cycles[data->count]);
		  data->ids[data->count] = data->count;
/* 		  if (options->verbose) */
/* 		    { */
/* 		      fprintf (options->out_file, "*"); */
/* 		      fflush (options->out_file); */
/* 		    } */
		  if (data->cycles[data->count] != 0)
		    {
		      data->count++;
		      ++tested;
		    }
		}
	      else
		++cache_hit;

	      // Go back.
	      int mybool = 1;
	      while (mybool)
		{
		  while (idx[j] >= 0 &&
			 Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]]))
		    {
		      Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
		      --(idx[j]);
		    }
		  if (idx[j] < 0)
		    {
		      idx[j] = 0;
		      --j;
		      if (j < 0)
			break;
		    }
		  else
		    mybool = 0;
		}
	      if (j < 0)
		{
		  printf ("Dimension terminated\n");
		  break;
		}
	      for (k = j; k < space->dimension; ++k)
		fm_vector_free (transfo[k]);
	    }
	  else
	    {
	      // Go to the next dimension.
	      ++j;
	    }
	}
      else
	{
	  ++(idx[j]);
	  fm_solver_compute_min
	    (&rat,
	     space->u_polyhedron[j]->solution[idx[j]].positive,
	     prototype[j], idx[j], FM_MINLEXICO_INT);
	  Z_ASSIGN(min[j][idx[j]], rat->num);
	  fm_solver_compute_max
	    (&rat,
	     space->u_polyhedron[j]->solution[idx[j]].negative,
	     prototype[j], idx[j], FM_MINLEXICO_INT);
	  Z_ASSIGN(max[j][idx[j]], rat->num);
	  Z_ASSIGN(ii[j][idx[j]], min[j][idx[j]]);
	  Z_DEC(ii[j][idx[j]], ii[j][idx[j]]);
	  while (idx[j] > 0 && (Z_CMP(min[j][idx[j]], >, max[j][idx[j]]) ||
				Z_CMP(ii[j][idx[j]], ==, max[j][idx[j]])))
	      --(idx[j]);
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "\n.... Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, ".... Number of cache hits: %d\n",
	       cache_hit);
    }

  // Be clean.
  for (j = 0; j < space->dimension; ++j)
    {
      for (i = 0; i < nb_iter; ++i)
	{
	  Z_CLEAR(ii[j][i]);
	  Z_CLEAR(min[j][i]);
	  Z_CLEAR(max[j][i]);
	}
      XFREE(ii[j]);
      XFREE(min[j]);
      XFREE(max[j]);
    }
  XFREE(ii);
  XFREE(min);
  XFREE(max);
  fm_rational_free (rat);
}



/**
 * H1 Exploration heuristic for multidimensional schedules.
 *
 * Principle: Try the range of iterator coefficients, with the closest
 * to 0 possible value for the other coefficients. Then, iterate on
 * constant with the best transformations previously found (margin of
 * 5% -- specified with option thresold).
 *
 */
void
ls_heuristic_h1m (s_ls_space_t* space,
		  CandlProgram* program,
		  s_ls_options_t* options)
{
  int i, j, tmp, dim;
  s_ls_explorer_t* data = XMALLOC(s_ls_explorer_t, 1);
  cycles_t orig = 0;
  GCOUNT = 0;
  data->size = MAX_CYCLES_SIZE;

  data->cycles = XMALLOC(cycles_t, data->size);
  data->u_draws = XMALLOC(s_fm_vector_t**, data->size);
  data->count = 0;
  data->ids = XMALLOC(unsigned, data->size);
  data->dimension = space->dimension;
  space->size = 0;


  // Allocate and initialize the prototype schedule.
  s_fm_vector_t** prototype = XMALLOC(s_fm_vector_t*, space->dimension);
  for (j = 0; j < space->dimension; ++j)
    {
      prototype[j] = fm_vector_alloc (space->u_polyhedron[j]->size + 1);
      for (i = 1; i < space->u_polyhedron[j]->size + 1; ++i)
	fm_vector_assign_int_idx (prototype[j], 0, i);
    }

  // Compile and execute the original code.
  if (options->verbose)
    {
      fprintf (options->out_file, "... Using H1-M heuristic\n");
      fprintf (options->out_file, ".... Compile and run the original code\n");
    }
  orig = ls_explorer_execorig (options);
  printf ("Orig: %llu\n", orig);

  // Look for good transformations in the iterator part.
  if (options->verbose)
    fprintf (options->out_file, ".... Explore iterator coefficients\n");
  ls_explorer_h1m_iterator (space, program, options, prototype, data);
  space->size = data->count;
  // Sort and thresold the transformations.
  ls_explorer_heuristic_filter_multi (options, data);

  // Look for good transformations in the parameter part.
  tmp = data->count;
  if (options->verbose)
    fprintf (options->out_file, ".... Explore parameter coefficients\n");
  ls_explorer_h1m_parameter (space, program, options, data);
  space->size += data->count - tmp;
  // Sort and thresold the transformations.
  ls_explorer_heuristic_filter_multi (options, data);

  // Look for good transformations in the constant part.
  tmp = data->count;
  if (options->verbose)
    fprintf (options->out_file, ".... Explore constant coefficients\n");
  ls_explorer_h1m_constant (space, program, options, data);
  space->size += data->count - tmp;
  // Sort and thresold the transformations.
  ls_explorer_heuristic_filter_multi (options, data);

  // Output results.
  if (options->verbose)
    {
      fprintf (options->out_file, ".... Original cycles: %lld\n", orig);
      fprintf (options->out_file, ".... Best transformation cycles: %lld\n",
	       data->cycles[0]);
      fprintf (options->out_file, ".... Best transformation Id: %d\n",
	       data->ids[0]);
      fprintf (options->out_file, "... Heuristic terminated\n");
    }

  // be clean.
  for (i = 0; i < space->dimension; ++i)
    fm_vector_free (prototype[i]);
  XFREE(prototype);
  XFREE(data->ids);
  XFREE(data->cycles);
  for (i = 0; i < data->count; ++i)
    {
      for (dim = 0; dim < space->dimension; ++dim)
	fm_vector_free (data->u_draws[i][dim]);
      XFREE(data->u_draws[i]);
    }
  XFREE(data->u_draws);
  XFREE(data);
}
