/*
 * heuristic-random.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/heuristic-random.h>

/**
 * Helper for random schedule generation, for one-dimensional
 * schedules. Integrate a cache to ensure to draw only distinct
 * schedules.
 *
 */
static
void
ls_random_lookup (s_ls_space_t* space,
		  CandlProgram* program,
		  s_ls_options_t* options,
		  s_ls_explorer_t* data)
{
  s_fm_solution_t* sol = space->polyhedron;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  int i;
  unsigned idx = 0;
  s_fm_vector_t* transfo;
  s_fm_vector_t* transfo_array[2]; transfo_array[1] = NULL;
  s_fm_vector_t* draw = fm_vector_alloc (space->polyhedron->size + 1);
  int cache_hit = 0;
  int value;
  int tested = 0;
  int rand_stop = data->count ? data->count * 100 : 10000;

  // Initialize the random number generator.
  srand (time (NULL));

  for (i = 0; i < options->rtries && rand_stop; ++i)
    {
      for (idx = 0; idx < sol->size; ++idx)
	{
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  if (Z_CMP(lb->num, >, Ub->num))
	    idx -= 2;
	  else
	    {
	      if (Z_CMP(lb->num, ==, Ub->num))
		value = lb->num;
	      else
		value = rand() % (Ub->num - lb->num) + lb->num;
	      fm_vector_assign_int_idx (draw, value, idx + 1);
	    }
	}
      // Ensure the transformation has not already been generated.
      if (! ls_explorer_transfo_exists (data, draw))
	{
	  transfo = fm_vector_dup (draw);
	  // Memorize and execute it.
	  data->draws[data->count] = transfo_array[0] = transfo;
	  data->cycles[data->count] =
	    ls_explorer_exectrans (program, options,
				   transfo_array, data->count);
	  data->ids[data->count] = data->count;
	  if (options->verbose)
	    {
	      fprintf (options->out_file, "*");
	      fflush (options->out_file);
	    }
	  if (data->cycles[data->count] != 0)
	    {
	      data->count++;
	      ++tested;
	    }
	}
      else
	{
	  ++cache_hit;
	  --i;
	  --rand_stop;
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "[LetSee] Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, "[LetSee] Number of cache hits: %d\n",
	       cache_hit);
    }

  // Be clean.
  fm_rational_free (lb);
  fm_rational_free (Ub);
  fm_vector_free (draw);
}



/**
 * Helper for random schedule generation, for multidimensional
 * schedules. Integrate a cache to ensure to draw only distinct
 * schedules.
 *
 */
static
void
ls_random_lookup_multi (s_ls_space_t* space,
			CandlProgram* program,
			s_ls_options_t* options,
			s_ls_explorer_t* data)
{
  s_fm_solution_t* sol;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  int i, dim;
  unsigned idx = 0;
  s_fm_vector_t** transfo;
  s_fm_vector_t* tmp;
  s_fm_vector_t** draw = XMALLOC(s_fm_vector_t*, space->dimension);
  int cache_hit = 0;
  int value;
  int tested = 0;
  int rand_stop = data->count ? data->count * 100 : 10000;

  // Initialize the output solution.
  for (i = 0; i < space->dimension; ++i)
    draw[i] = fm_vector_alloc (space->u_polyhedron[i]->size + 1);

  // Initialize the random number generator.
  srand (time (NULL));

  for (i = 0; i < options->rtries && rand_stop; ++i)
    {
      for (dim = 0; dim < space->dimension; ++dim)
	{
	  sol = space->u_polyhedron[dim];
	  for (idx = 0; idx < sol->size; ++idx)
	    {
	      fm_solver_compute_min (&lb, sol->solution[idx].positive,
				     draw[dim], idx, FM_MINLEXICO_INT);
	      fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				     draw[dim], idx, FM_MINLEXICO_INT);
	      // Avoid integer holes.
	      ls_heuristics_narrow_intbounds (lb, Ub, sol, idx, draw[dim]);
	      
	      if (Z_CMP(lb->num, ==, Ub->num))
		value = lb->num;
	      else
		value = rand() % (Ub->num - lb->num) + lb->num;
	      fm_vector_assign_int_idx (draw[dim], value, idx + 1);
	    }

	  // Ensure correctness.
	  ls_heuristics_fixcomplete (options, space, draw, dim);
	}
      // Ensure the transformation has not already been generated.
      if (! ls_explorer_transfo_exists_multi (data, draw))
	{
	  // Memorize it.
	  data->u_draws[data->count] =
	    XMALLOC(s_fm_vector_t*, space->dimension + 1);
	  data->u_draws[data->count][space->dimension] = NULL;
	  for (dim = 0; dim < space->dimension; ++dim)
	    data->u_draws[data->count][dim] = fm_vector_dup (draw[dim]);

	  // Execute it.
	  data->cycles[data->count] =
	    ls_explorer_exectrans (program, options,
				   data->u_draws[data->count],
				   data->count);
	  data->ids[data->count] = data->count;
	  if (options->verbose && options->pocc_utils == NULL)
	    {
	      fprintf (options->out_file, "*");
	      fflush (options->out_file);
	    }
	  if (data->cycles[data->count] != 0)
	    {
	      data->count++;
	      ++tested;
	    }
	}
      else
	{
	  ++cache_hit;
	  --i;
	  --rand_stop;
	}
    }

  if (options->verbose)
    {
      fprintf (options->out_file, "[LetSee] Number of tested schedules: %d\n",
	       tested);
      fprintf (options->out_file, "[LetSee] Number of cache hits: %d\n",
	       cache_hit);
    }

  // Be clean.
  fm_rational_free (lb);
  fm_rational_free (Ub);
  for (i = 0; i < space->dimension; ++i)
    fm_vector_free (draw[i]);
  XFREE(draw);
}




/**
 * Random Exploration heuristic R1.
 *
 * Principle: Try randomly X transformations.
 *
 */
void
ls_heuristic_random_r1 (s_ls_space_t* space,
			CandlProgram* program,
			s_ls_options_t* options)
{
  int i, tmp;
  s_ls_explorer_t* data = XMALLOC(s_ls_explorer_t, 1);
  cycles_t orig = 0;

  data->size = MAX_CYCLES_SIZE;

  data->cycles = XMALLOC(cycles_t, data->size);
  data->draws = XMALLOC(s_fm_vector_t*, data->size);
  data->count = 0;
  data->ids = XMALLOC(unsigned, data->size);

  space->size = 0;

  if (options->verbose)
    {
      fprintf (options->out_file, "... Using Random heuristic R1\n");
      fprintf (options->out_file, ".... Compile and run the original code\n");
    }

  // Compile and execute the original code.
  orig = ls_explorer_execorig (options);

  if (options->verbose)
    fprintf (options->out_file, ".... First random exploration\n");
  // Randomly test some points.
  ls_random_lookup (space, program, options, data);
  space->size = data->count;

  for (i = 0; i < data->count; ++i)
    printf ("%d %Ld\n", i, data->cycles[i]);

  // Sort and thresold the transformations.
  ls_explorer_heuristic_filter (options, data);

  // Output results.
  if (options->verbose)
    {
      fprintf (options->out_file, ".... Original cycles: %lld\n", orig);
      fprintf (options->out_file, ".... Best transformation cycles: %lld\n",
	       data->cycles[0]);
      fprintf (options->out_file, ".... Best transformation Id: %d\n",
	       data->ids[0]);
      fprintf (options->out_file, "... Heuristic terminated\n");
    }

  // be clean.
  XFREE(data->ids);
  XFREE(data->cycles);
  for (i = 0; i < data->count; ++i)
    fm_vector_free (data->draws[i]);
  XFREE(data->draws);
  XFREE(data);
}


/**
 * Random Exploration heuristic R1M, designed for multidimensional
 * schedules.
 *
 * Principle: Try randomly X transformations.
 *
 */
void
ls_heuristic_random_r1m (s_ls_space_t* space,
			 CandlProgram* program,
			 s_ls_options_t* options)
{
  int i, j, tmp;
  s_ls_explorer_t* data = XMALLOC(s_ls_explorer_t, 1);
  cycles_t orig = 0;

  data->size = MAX_CYCLES_SIZE;

  data->cycles = XMALLOC(cycles_t, data->size);
  data->u_draws = XMALLOC(s_fm_vector_t**, data->size);
  data->count = 0;
  data->ids = XMALLOC(unsigned, data->size);
  data->dimension = space->dimension;

  space->size = 0;

  if (options->verbose)
    {
      fprintf (options->out_file, "[LetSee] Using Random heuristic R1-multi\n");
      if (options->compile_line != NULL)
	fprintf (options->out_file, 
	       "[LetSee] Compile and run the original code\n");
    }

  // Compile and execute the original code.
  if (options->compile_line != NULL)
    orig = ls_explorer_execorig (options);

  // Randomly test some points.
  ls_random_lookup_multi (space, program, options, data);
  space->size = data->count;

  if (options->pocc_utils == NULL)
    for (i = 0; i < data->count; ++i)
      printf ("%d %Ld\n", i, data->cycles[i]);

  // Sort and thresold the transformations.
  ls_explorer_heuristic_filter_multi (options, data);

  // Output results.
  if (options->verbose)
    {
      if (options->pocc_utils == NULL)
	fprintf (options->out_file, "[LetSee] Original cycles: %lld\n", orig);
      fprintf (options->out_file, "[LetSee] Best transformation cycles: %lld\n",
	       data->cycles[0]);
      fprintf (options->out_file, "[LetSee] Best transformation Id: %d\n",
	       data->ids[0]);
      fprintf (options->out_file, "[LetSee] Heuristic terminated\n");
    }

  // be clean.
  XFREE(data->ids);
  XFREE(data->cycles);
  for (i = 0; i < data->count; ++i)
    {
      for (j = 0; j < space->dimension; ++j)
	fm_vector_free (data->u_draws[i][j]);
      XFREE(data->u_draws[i]);
    }
  XFREE(data->u_draws);
  XFREE(data);
}

