/*
 * lspace.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/lspace.h>


/*
 * @FIXME: This unit is kept for backward compatibility reason.
 *
 *
 */


#define LSPACE_REDREC_DESCENDANT 1
#define LSPACE_REDREC_IRIGOIN    2

#define echo(S) { printf(#S); printf("\n"); }



/**
 * \brief Initialize the solution polytope with predefined bounds
 * provided in `options' parameter.
 *
 * The solution polytope is formatted w.r.t. the [ i1 i2 p1 p2 c1 c2 ]
 * organization.
 *
 */
static
s_fm_solution_t*
ls_lspace_initialize_solution (CandlProgram* program,
			       s_ls_options_t* options)
{
  int i, j;

  // Compute the size of the polytope.
  int sz = 0;
  for (i = 0; i < program->nb_statements; ++i)
    sz += program->statement[i]->domain->NbColumns - 1;
  int nb_param = program->statement[0]->domain->NbColumns -
    program->statement[0]->depth - 2;

  // Allocate the solution polytope.
  s_fm_solution_t* P = fm_solution_alloc (sz);

  // Fill the solution polytope with the initial bounds.
  int count = 1;
  s_fm_vector_t* tmp;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  z_type_t mlb; Z_INIT(mlb); Z_ASSIGN_SI(mlb, options->lb); Z_OPP(mlb, mlb);

  // Deal with iterator coefficients.
  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < program->statement[i]->depth; ++j)
      {
	// Add iUb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, mone, count);
	fm_vector_assign_int_idx (tmp, options->Ub, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	// Add ilb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, one, count);
	fm_vector_assign_int_idx (tmp, mlb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	++count;
      }

  // Deal with parameter coefficients.
  Z_ASSIGN_SI(mlb, options->plb); Z_OPP(mlb, mlb);
  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < nb_param; ++j)
      {
	// Add pUb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, mone, count);
	fm_vector_assign_int_idx (tmp, options->pUb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	// Add plb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, one, count);
	fm_vector_assign_int_idx (tmp, mlb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	++count;
      }

  // Deal with constant coefficients.
  Z_ASSIGN_SI(mlb, options->clb); Z_OPP(mlb, mlb);
  for (i = 0; i < program->nb_statements; ++i)
    {
      // Add cUb.
      tmp = fm_vector_alloc (count + 2);
      fm_vector_assign_int_idx (tmp, mone, count);
      fm_vector_assign_int_idx (tmp, options->cUb, count + 1);
      fm_vector_set_ineq (tmp);
      fm_solution_add_line_at (P, tmp, count);
      // Add clb.
      tmp = fm_vector_alloc (count + 2);
      fm_vector_assign_int_idx (tmp, one, count);
      fm_vector_assign_int_idx (tmp, mlb, count + 1);
      fm_vector_set_ineq (tmp);
      fm_solution_add_line_at (P, tmp, count);
      ++count;
    }

  Z_CLEAR(one);
  Z_CLEAR(mone);
  Z_CLEAR(mlb);

  return P;
}



/**
 * \brief Compute the system to solve, for a given dependence and a
 * pair of statements. Outputs the number of schedule coefficients in
 * the system (int* dimension).
 *
 * The function computes the set of Farkas multipliers for the dependence
 * polyhedron (previously computed with Candl), and then equate them
 * with the schedule coefficients of the statement(s). There is
 * exactly one equality per column in the dependence polyhedron. The
 * set of inequality corresponds to the positivity constraints on the
 * Farkas multipliers.
 *
 * The system is of the form [ i1 i2 p1 p2 c1 c2 lambda ].
 *
 * @fixme The coefficient constraints are not mandatory, but seem to
 * help the resolution to be faster.
 *
 */
static
s_fm_system_t*
ls_lspace_build_system (CandlProgram* program,
			s_ls_options_t* options,
			CandlDependence* tmp,
			int* dimension,
			s_fm_solution_t* P,
			int opp,
			int type)
{
  int i, j, k;

  // Compute the number of Farkas multipliers for the dependence (plus
  // lambda 0).
  int sz_lambda = 1;
  for (i = 0; i < tmp->domain->NbRows; ++i, ++sz_lambda)
    if (Z_CMP_SI(tmp->domain->p[i][0], ==, 0))
      ++sz_lambda;

  // Compute the number of schedule coefficients related to the dependence.
  *dimension = tmp->source->domain->NbColumns - 1;
  if (tmp->source !=  tmp->target)
    *dimension += tmp->target->domain->NbColumns - 1;

  // Allocate the system.
  s_fm_system_t* system =
    fm_system_alloc (sz_lambda + tmp->domain->NbColumns - 1
		     + 2 * *dimension, sz_lambda + *dimension + 2);

  // System is: t_Target - t_Source = Lambda.
  //
  // Affect the "first" statement to st1, the "second" to st2.
  CandlStatement* st1;
  CandlStatement* st2;
  CandlStatement* stmp;
  z_type_t val; Z_INIT(val);
  z_type_t modifier; Z_INIT(modifier);
  if (tmp->source->label <= tmp->target->label)
    {
      st1 = tmp->source;
      st2 = tmp->target;
      Z_ASSIGN_SI(modifier, 1);
    }
  else
    {
      st2 = tmp->source;
      st1 = tmp->target;
      Z_ASSIGN_SI(modifier, -1);
    }

  if (opp)
    {
      stmp = st1;
      st1 = st2;
      st2 = stmp;
      Z_ASSIGN_SI(val, -1);
      Z_MUL(modifier, modifier, val);
    }

  int count = 0;
  int lambda_count = 0;
  int l_offs = 0;
  if (Z_CMP_SI(modifier, ==, -1))
    l_offs = st2->depth;
  // Fill the st1 iterator part.
  Z_ASSIGN_SI(val, -1);
  Z_MUL(val, val, modifier);
  for (i = 1; i <= st1->depth; ++i)
    {
      fm_vector_assign_int_idx (system->lines[count], val, i);
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i + l_offs],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	    fm_vector_assign_int_idx (system->lines[count],
				      tmp->domain->p[j][i + l_offs],
				      ++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

/*   fm_vector_assign_int_idx (system->lines[0], val, system->nb_cols - 1); */


  // Fill the st2 iterator part.
  Z_OPP(val, val);
  if (Z_CMP_SI(modifier, ==, -1))
    l_offs = st1->depth;
  for (; i <= st1->depth + st2->depth; ++i)
    {
      if (st1 != st2)
	{
	  fm_vector_assign_int_idx (system->lines[count], val, i);
/*     fm_vector_assign_int_idx (system->lines[count], val, system->nb_cols - 1); */
	}
      else
	{
	  fm_vector_assign_int_idx (system->lines[count], val, i - st1->depth);
/*     fm_vector_assign_int_idx (system->lines[count], val, system->nb_cols - 1); */
	}
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i - l_offs],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	    fm_vector_assign_int_idx (system->lines[count],
				      tmp->domain->p[j][i - l_offs],
				      ++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

  // Fill the st1 and st2 parameter part.
  Z_OPP(val, val);
  int sz_param = st1->domain->NbColumns - st1->depth - 2;
  for (; i <= st1->depth + st2->depth + sz_param; ++i)
    {
      if (st1 != st2)
	{
	  fm_vector_assign_int_idx (system->lines[count], val, i);
	  Z_OPP(val, val);
	  fm_vector_assign_int_idx (system->lines[count], val, i + sz_param);
	  Z_OPP(val, val);
	}
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	      fm_vector_assign_int_idx (system->lines[count],
					tmp->domain->p[j][i],
					++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

  // Fill the st1 and st2 constant part.
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, -1);
  // Add -lambda_0, and -1.
  if (type == LS_LSPACE_BUILD_GEONE)
    fm_vector_assign_int_idx (system->lines[count], one, system->nb_cols - 1);
  fm_vector_assign_int_idx (system->lines[count], one, *dimension + 1);
  if (st1 != st2)
    {
      fm_vector_assign_int_idx (system->lines[count], val, i + sz_param);
      Z_OPP(val, val);
      fm_vector_assign_int_idx (system->lines[count], val, i + sz_param + 1);
      Z_OPP(val, val);
    }
  lambda_count = *dimension + 2;
  for (j = 0; j < tmp->domain->NbRows; ++j)
    {
      fm_vector_assign_int_idx (system->lines[count],
				tmp->domain->p[j][i],
				lambda_count);
      Z_OPP(system->lines[count]->vector[lambda_count].num,
	    system->lines[count]->vector[lambda_count].num);
      if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	fm_vector_assign_int_idx (system->lines[count],
				  tmp->domain->p[j][i],
				  ++lambda_count);
      ++lambda_count;
    }
  ++count;

  // Add positivity constraints for the Farkas multipliers.
  Z_OPP(one, one);
  for (i = *dimension + 1; i < system->nb_cols - 1; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count++], one, i);
    }

  // Add coefficients constraints.
  // FIXME: This is not necessary.
  int bound;
  z_type_t lb; Z_INIT(lb); Z_ASSIGN(lb, options->lb); Z_OPP(lb, lb);
  z_type_t Ub; Z_INIT(Ub); Z_ASSIGN(Ub, options->Ub);

  // Copy iterator constraints.
  if (st1 != st2)
    bound = st1->depth + st2->depth;
  else
    bound = st1->depth;
  for (i = 1; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Copy parameter constraints.
  Z_ASSIGN(lb, options->plb); Z_OPP(lb, lb);
  Z_ASSIGN(Ub, options->pUb);
  if (st1 != st2)
    bound = st1->depth + st2->depth + 2 * sz_param;
  else
    bound = st1->depth + sz_param;
  for (; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Copy constant constraints.
  Z_ASSIGN(lb, options->clb); Z_OPP(lb, lb);
  Z_ASSIGN(Ub, options->cUb);
  if (st1 != st2)
    bound += 2;
  else
    bound += 1;
  for (; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Be clean.
  Z_CLEAR(modifier);
  Z_CLEAR(val);
  Z_CLEAR(one);

  return system;
}



/**
 * \brief Do a clever intersection of the solution obtained for the
 * dependence `dep' with the solution polytope `P'. Try to preserve
 * FM-property as much as possible.
 *
 *
 *
 */
static
void
ls_lspace_intersect (CandlProgram* program,
		     s_ls_options_t* options,
		     CandlDependence* dep,
		     s_fm_solution_t* P,
		     s_fm_solution_t* s,
		     int opp)
{
  s_fm_list_t* tmp;
  CandlStatement* s1;
  CandlStatement* s2;
  CandlStatement* stmp;
  s_fm_vector_t* v;
  s_fm_vector_t* c;
  int i, j;
  int bound, P_idx;
  int i_start;


  // Affect the "first" statement to st1, the "second" to st2.
  if (dep->source->label < dep->target->label)
    {
      s1 = dep->source;
      s2 = dep->target;
    }
  else
    {
      s2 = dep->source;
      s1 = dep->target;
    }

  // Copy the s1 iterator part.
  //
  // Compute i1 offset.
  int i1_offset = 0;
  for (i = 0; i < s1->label; ++i)
    i1_offset += program->statement[i]->depth;

  // Copy, for all i of s1.
  bound = s1->depth;
  for (i = 0; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }


  // Copy the s2 iterator part.
  //
  // Compute i2 offset.
  int i2_offset = 0;
  for (j = 0; j < s2->label; ++j)
    i2_offset += program->statement[j]->depth;

  if (s1 != s2)
    {
      bound += s2->depth;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 parameter part.
  //
  // Compute the number of parameters, and the p1 offset.
  int nb_param = program->statement[0]->domain->NbColumns - 2
    - program->statement[0]->depth;
  int p1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p1_offset += program->statement[j]->depth;
  for (j = 0; j < s1->label; ++j)
    p1_offset += nb_param;

  bound += nb_param;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 parameter part.
  //
  // Compute the p2 offset.
  int p2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p2_offset += program->statement[j]->depth;
  for (j = 0; j < s2->label; ++j)
    p2_offset += nb_param;

  if (s1 != s2)
    {
      bound += nb_param;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 constant part.
  //
  // Compute the c1 offset.
  int c1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c1_offset += program->statement[j]->depth;
  c1_offset += nb_param * program->nb_statements;
  c1_offset += s1->label;

  bound += 1;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 constant part.
  //
  // Compute the c2 offset.
  int c2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c2_offset += program->statement[j]->depth;
  c2_offset += nb_param * program->nb_statements;
  c2_offset += s2->label;

  if (s1 != s2)
    {
      bound += 1;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }
}



/**
 * \brief Compute the solution polytope, given a set of dependences, program
 * information and options provided to LetSee.
 *
 * The computation depends on:
 * - the initalization of the solution polytope,
 * - the construction of the system to solve,
 * - the intersection of the obtained solution set with the solution
 *   polytope.
 *
 */
static
s_fm_solution_t*
ls_lspace_compute_solution (CandlDependence* dependences,
			    CandlProgram* program,
			    s_ls_options_t* options)
{
  s_fm_solution_t* PA[10];
  s_fm_solution_t* P;

  s_fm_solution_t* Ptmp;

  s_fm_system_t* Stmp;
  s_fm_system_t* to_solve;
  s_fm_solution_t* solution;
  int d_count = 0, dimension;
  CandlDependence* tmp;
  int count = 0;

  int ii;

  int unresolved = 1;

/*   while (unresolved) */
    {
      // Initialize the solution polytope.
      P = ls_lspace_initialize_solution (program, options);
      unresolved = 0;
      // Loop on each dependence.
      for (tmp = dependences; tmp != NULL; tmp = tmp->next)
	{
	  // Skip the dependence if it has already been solved.
/* 	  if (tmp->solved) */
/* 	    continue; */

	  if (options->verbose)
	    fprintf (options->out_file,
		     ".... Solving dependence %d\n", ++d_count);
/* 	  pip_matrix_print (stdout, tmp->domain); */

	  // Build the system to solve, for the given dependence.
	  to_solve = ls_lspace_build_system (program, options, tmp,
					     &dimension, P, 0,
					     LS_LSPACE_BUILD_GEONE);

	  // Solve the system.
/* 	  fm_system_print (stdout, to_solve); */

/* 	  printf ("Polyhedron check: "); */
/* 	  if (fm_piptools_check_rat (to_solve)) */
/* 	    printf ("has point\n"); */
/* 	  else */
/* 	    printf ("no point\n"); */
/* 	  continue; */
	  solution = fm_solver_solution_at //  | FM_SOLVER_REDREC_IRIGOIN
	    (to_solve, FM_SOLVER_FAST, dimension);
/* 	  fm_solution_print (stdout, solution); */

	  // fm_piptools_polyhedron_check (NULL, options, solution);
	  Ptmp = fm_solution_dup (P);

	  // Add the obtained solution set to the solution polytope.
	  ls_lspace_intersect (program, options, tmp, P, solution, 0);

	  // Be clean.
	  fm_system_free (to_solve);
	  fm_solution_free (solution);

	  // fm_piptools_polyhedron_check (NULL, options, P);

	  Stmp = fm_solution_to_system (P);
	  Stmp = fm_system_to_z (Stmp);

	  if (! fm_piptools_check_int (Stmp))
	    {
	      if (options->verbose)
		{
		  fprintf (options->out_file,
			   ".... No one-dimensional solution. Please consider '-t multi' option.\n");
		  exit (1);
		}
	      unresolved = 1;
	      // No point.
	      fm_solution_free (P);
	      P = Ptmp;
	      // Build the system to solve, for the given dependence.
	      to_solve = ls_lspace_build_system (program, options, tmp,
						 &dimension, P, 0,
						 LS_LSPACE_BUILD_EQZERO);
	      // Solve the system.
	      solution = fm_solver_solution_at //  | FM_SOLVER_REDREC_IRIGOIN
		(to_solve, FM_SOLVER_FAST, dimension);
	      // Add the obtained solution set to the solution polytope.
	      ls_lspace_intersect (program, options, tmp, P, solution, 0);
	      // Be clean.
	      fm_system_free (to_solve);
	      fm_solution_free (solution);

	      fm_piptools_check_sol (P, FM_PIPTOOLS_INT);
	      CANDL_DEP_TAG(tmp, solved) = 0;
	    }
	  else
	    {
	      fm_solution_free (Ptmp);
/* 	      tmp->solved = 1; */
	    }
	  fm_system_free (Stmp);

	  /*       // Build the system to solve, for the given dependence. */
	  /*       to_solve = ls_lspace_build_system (program, options, tmp, */
	  /* 					 &dimension, P, 1); */
	  /*       // Solve the system. */
	  /*       solution = fm_solver_solution_at (to_solve, FM_SOLVER_FAST, dimension); */
	  /*       // Add the obtained solution set to the solution polytope. */
	  /*       ls_lspace_intersect (program, options, tmp, P, solution, 1); */
	  /*       // Be clean. */
	  /*       fm_system_free (to_solve); */
	  /*       fm_solution_free (solution); */

	}
      PA[count++] = P;
    }

  return P;
}



/**
 * \brief order the dependence graph.
 *
 *
 *
 *
 */
static
CandlDependence*
ls_lspace_depgraph_preprocess (CandlProgram* program,
			       CandlDependence* dependences)
{
  int i, j, size = 0;
  CandlDependence* head = dependences;
  CandlDependence* tmp;

  // Initialize the `prev' record in the CandlDependence structure.
  for (tmp = dependences; tmp != NULL; tmp = tmp->next, ++size)
    ;

  // Copy data into a temporary array.
  CandlDependence** tab = XMALLOC(CandlDependence*, size);
  for (i = 0; i < size; ++i, head = head->next)
    tab[i] = head;

  // Sort the graph.
  for (i = 0; i < size; ++i)
    for (j = i; j < size; ++j)
      if (tab[i]->source->label < tab[j]->source->label ||
	  (//tab[i]->source->label == tab[j]->source->label &&
	   tab[i]->target->label < tab[j]->target->label
	   || tab[i]->source->label < tab[j]->target->label
	   || tab[i]->target->label < tab[j]->source->label
	   ))
	{
	  tmp = tab[i];
	  tab[i] = tab[j];
	  tab[j] = tmp;
	}

  // Finalize the sorted list.
  head = tmp = tab[0];
  for (i = 0; i < size - 1; ++i, tmp->next = tab[i], tmp = tmp->next)
    ;
  tmp->next = NULL;

  // Be clean.
  XFREE(tab);

  return head;
}


/**
 * \brief Build the legal scheduling space, for one-dimensional affine
 * schedules.
 *
 * The output solution is represented by the polytope P, organized as follows:
 * For s1 = [i1 p1 c1] the schedule of the first statement (i1 is the
 * "iterator" part of the schedule, p1 the parameter part and c1 is
 * the constant one), and respectivly s2 = [i2 p2 c2] for the second statement;
 * the output polytope P is of the form P = [ i1 i2 p1 p2 c1 c2 ].
 *
 *
 * See the Master Thesis "Legal transformation space exploration in
 * the polyhedral model: the monodimensional affine scheduling case",
 * LN Pouchet, 2006.
 *
 */
s_ls_space_t*
ls_lspace_build (CandlProgram* program,
		 CandlDependence* dependences,
		 s_ls_options_t* options)
{
  if (options->verbose)
    fprintf(options->out_file, "... Build and solve local systems\n");

  // Preprocess the dependence graph.
  CandlDependence* graph = dependences;
/*     ls_lspace_depgraph_preprocess (program, dependences); */
  candl_dependence_pprint (stdout, graph);

  // Compute the solution polytope.
  s_fm_solution_t* P =
    ls_lspace_compute_solution (graph, program, options);

  if (options->verbose)
    fprintf(options->out_file, "... All systems solved\n");

  // Initial PIP check.
  if (! fm_piptools_check_sol (P, FM_PIPTOOLS_INT))
    {
      if (options->verbose)
	fprintf(options->out_file, "=> Ending computation\n");
      exit (1);
    }

  // Allocate and fill the output.
  s_ls_space_t* space = ls_space_alloc ();
  space->polyhedron = P;

  if (options->normalize_space)
    {
      P = fm_solution_simplify
	(P, FM_SOLVER_REDREC_IRIGOIN | FM_SOLVER_VERBOSE);
      s_fm_system_t* system = fm_solution_to_system (P);
      space->polyhedron =
	fm_solver (system, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN);
      fm_solution_free (P);
      fm_system_free (system);
      if (options->verbose)
	fprintf(options->out_file, "... Solution normalized\n");
    }

  space->dimension = 1;
  space->u_polyhedron = XMALLOC(s_fm_solution_t*, 1);
  space->u_polyhedron[0] = space->polyhedron;

/*   s_fm_system_t* s = fm_solution_to_system (space->polyhedron); */
/*   s_fm_system_t* sz = fm_system_to_z (s); */
/*   fm_system_free (s); */

/*   fm_system_print (stdout, sz); */
/*   fm_system_free (sz); */

  return space;
}
