/*
 * transformations.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/space.h>
#include <letsee/transformations.h>
#include <candl/candl.h>

int TGCOUNT = 0;

static
int		ls_output_file (FILE* file,
			        CandlProgram* program,
				PipMatrix** schedules)
{
  unsigned	i, j, k;

  // Header.
  fprintf (file, "# ---------------------- CONTEXT ----------------------\n");
  // Target language.
  fprintf (file, "c # Language is C.\n");

  // Context.
  fprintf (file, "# Context: \n");
  pip_matrix_print (file, program->context);

  // Parameters are automatically named.
  fprintf (file, "0 # parameters are automatically named.\n");
  fprintf (file, "\n\n");

  fprintf (file, "# --------------------- STATEMENTS --------------------\n");
  // Print statements.
  fprintf (file, "%d  # Number of statements.\n", program->nb_statements);

  for (i = 0; i < program->nb_statements; ++i)
    {
      // At the moment, disjoint domain are not handled.
      fprintf (file, "1 # Number of domain polyhedron for statement %d.\n",
	       i + 1);
      pip_matrix_print (file, program->statement[i]->domain);
      // Must add "0 0 0" at the end of the domain matrix.
      fprintf (file, "0 0 0 # For future options.\n");
      fprintf (file, "\n\n");
    }

  // Iterator are automatically named.
  fprintf (file, "0 # Iterators are automatically named.\n");
  fprintf (file, "\n\n");

  // Print scattering functions.
  fprintf (file, "# --------------------- SCATTERING --------------------\n");
  fprintf (file, "%d  # Number of scattering functions.\n",
	   program->nb_statements);
  for (i = 0; i < program->nb_statements; ++i)
    // Print the (CLooG formated) schedule.
    pip_matrix_print (file, schedules[i]);
  fprintf (file, "0 # Scattering indexes are automatically named.\n");
  fprintf (file, "\n");

  // Violation check.

  // Avoid the check if we have one-dimensional schedules (bug in candl)
  if (schedules[0]->NbRows > 1)
    {
      CandlOptions* options = candl_options_malloc ();
      // options->fullcheck = 1;
      PipMatrix* sched[program->nb_statements];
      for (i = 0; i < program->nb_statements; ++i)
	{
	  sched[i] = pip_matrix_alloc (schedules[i]->NbRows,
				       schedules[i]->NbColumns -
				       schedules[i]->NbRows);
	  for (j = 0; j < schedules[i]->NbRows; ++j)
	    {
	      Z_ASSIGN_SI(sched[i]->p[j][0], 1);
	      for (k = 1; k < sched[i]->NbColumns; ++k)
		{
		  Z_ASSIGN(sched[i]->p[j][k],
			   schedules[i]->p[j][k + schedules[i]->NbRows]);
		  // Z_OPP(sched[i]->p[j][k], sched[i]->p[j][k]);
		}
	    }
	}
/*       return 0; */

      program->transformation = sched;
/*       CandlViolation* v = candl_violation (program, NULL, options); */
      CandlViolation* v = NULL;
      candl_options_free (options);
      /*   for (i = 0; i < program->nb_statements; ++i) */
      /*     pip_matrix_free (sched[i]); */
      program->transformation = NULL;

      TGCOUNT++;
/*       printf ("transfo was tested for violation\n"); */
      if (v)
	{
	  printf ("VIOLATION\n");

	  candl_violation_pprint(stdout, v);
	  for (i = 0; i < program->nb_statements; ++i)
	    pip_matrix_print (stdout, sched[i]);

	  exit (72);

	  candl_violation_free (v);
	  return 0;
	}
      else
	{
/* 	  printf ("No violation at %d\n", TGCOUNT - 1); */
	  /*       for (i = 0; i < program->nb_statements; ++i) */
	  /* 	pip_matrix_print (stdout, sched[i]); */
	}
    }

  return 0;
}



/**
 * \brief convert a point in the legal schedules polytope into a CLooG
 * formatted set of matrices, based on a polytope of the form:
 * P = [ i1 p1 c1 i2 p2 c2 ].
 *
 *
 */
static
PipMatrix** ls_transfo_to_cloog (CandlProgram* program,
				 s_fm_vector_t* draw)
{
  unsigned i, j;
  unsigned offset;
  PipMatrix** res = XMALLOC(PipMatrix*, program->nb_statements);

  for (i = 0, offset = 0; i < program->nb_statements; ++i)
    {
      res[i] =
	pip_matrix_alloc (1, program->statement[i]->domain->NbColumns + 1);
      Z_ASSIGN_SI(res[i]->p[0][1], -1);
      for (j = 1; j < program->statement[i]->domain->NbColumns; ++j)
	Z_ASSIGN(res[i]->p[0][j + 1], draw->vector[offset + j].num);

      offset += j - 1;
    }

  return res;
}



/**
 * \brief convert a point in the legal schedules polytope into a CLooG
 * formatted set of matrices, based on a polytope of the form:
 * P = [ i1 i2 p1 p2 c1 c2 ].
 *
 *
 */
static
PipMatrix** ls_transfo_to_cloog_ospace (CandlProgram* program,
					s_fm_vector_t* draw)
{
  unsigned i, j, jj;
  unsigned offset;
  unsigned pos;
  unsigned nb_param = program->statement[0]->domain->NbColumns - 2 -
    program->statement[0]->depth;
  PipMatrix** res = XMALLOC(PipMatrix*, program->nb_statements);

  for (i = 0; i < program->nb_statements; ++i)
    {
      pos = 2;
      res[i] =
	pip_matrix_alloc (1, program->statement[i]->domain->NbColumns + 1);
      Z_ASSIGN_SI(res[i]->p[0][1], -1);
      // Copy the iterator part.
      offset = 1;
      for (j = 0; j < i; ++j)
	offset += program->statement[j]->depth;
      for (jj = 0; jj < program->statement[i]->depth; ++jj)
	Z_ASSIGN(res[i]->p[0][pos++], draw->vector[offset + jj].num);
      // Copy the parameter part.
      for (; j < program->nb_statements; ++j)
	offset += program->statement[j]->depth;
      for (j = 0; j < i; ++j)
	offset += nb_param;
      for (jj = 0; jj < nb_param; ++jj)
	Z_ASSIGN(res[i]->p[0][pos++], draw->vector[offset + jj].num);
      // Copy the constant part.
      for (; j < program->nb_statements; ++j)
	offset += nb_param;
      for (j = 0; j < i; ++j)
	offset += 1;
      Z_ASSIGN(res[i]->p[0][pos++], draw->vector[offset].num);
    }

  return res;
}



/**
 * \brief convert a point in the legal schedules polytope into a CLooG
 * formatted set of matrices, based on a polytope of the form:
 * P = [ i1 i2 p1 p2 c1 c2 ].
 *
 * Multidimensional version.
 *
 *
 */
static
PipMatrix** ls_transfo_to_cloog_multi (CandlProgram* program,
				       s_fm_vector_t** draw)
{
  unsigned i, j, jj, k;
  unsigned offset;
  unsigned pos;
  unsigned nb_param = program->statement[0]->domain->NbColumns - 2 -
    program->statement[0]->depth;
  PipMatrix** res = XMALLOC(PipMatrix*, program->nb_statements);
  int dim;
  for (dim = 0; draw[dim] != NULL; ++dim)
    ;

  for (i = 0; i < program->nb_statements; ++i)
    {
      res[i] =
	pip_matrix_alloc (dim, program->statement[i]->domain->NbColumns + dim);
      for (k = 0; k < dim; ++k)
	{
	  pos = dim + 1;
	  Z_ASSIGN_SI(res[i]->p[k][k + 1], -1);
	  // Copy the iterator part.
	  offset = 1;
	  for (j = 0; j < i; ++j)
	    offset += program->statement[j]->depth;
	  for (jj = 0; jj < program->statement[i]->depth; ++jj)
	    Z_ASSIGN(res[i]->p[k][pos++], draw[k]->vector[offset + jj].num);
	  // Copy the parameter part.
	  for (; j < program->nb_statements; ++j)
	    offset += program->statement[j]->depth;
	  for (j = 0; j < i; ++j)
	    offset += nb_param;
	  for (jj = 0; jj < nb_param; ++jj)
	    Z_ASSIGN(res[i]->p[k][pos++], draw[k]->vector[offset + jj].num);
	  // Copy the constant part.
	  for (; j < program->nb_statements; ++j)
	    offset += nb_param;
	  for (j = 0; j < i; ++j)
	    offset += 1;
	  Z_ASSIGN(res[i]->p[k][pos++], draw[k]->vector[offset].num);
	}
    }

  return res;
}



/**
 * \brief Create a CLooG formatted file for a given point in the legal
 * polytope.
 *
 *
 */
int
ls_transformations_generate (s_fm_vector_t* draw,
			     CandlProgram* program,
			     s_ls_options_t* options,
			     unsigned count)
{

  if (! options->create_schedfiles)
    return 0;

  char			str[100];
  char			buff[10];
  FILE*			t_file;
  unsigned		i;
  PipMatrix**		schedules;
  int			ret;

  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buff, "%09d", count);
  strcat(str, buff);
  strcat(str, ".cloog");

  t_file = fopen (str, "w");

  if (options->type == LS_TYPE_OSPACE)
    schedules = ls_transfo_to_cloog_ospace (program, draw);
  else
    schedules = ls_transfo_to_cloog (program, draw);
  ret = ls_output_file(t_file, program, schedules);

  // Be clean.
  for (i = 0; i < program->nb_statements; ++i)
    pip_matrix_free (schedules[i]);
  XFREE(schedules);
  fclose (t_file);

  return ret;
}


PipMatrix**
ls_transformations_tile_schedules (CandlProgram* program,
				   PipMatrix** schedules)
{
  PipMatrix** ret = XMALLOC(PipMatrix*, program->nb_statements);
  int i;
  int j;
  int k;

  for (i = 0; i < program->nb_statements; ++i)
    {
      ret[i] = pip_matrix_alloc (schedules[i]->NbRows + 1,
				 schedules[i]->NbColumns + 2);
      // Add the new dimension.
      ret[i]->p[0][0] = 0;
      ret[i]->p[0][1] = -1;
      ret[i]->p[0][schedules[i]->NbRows + 2] = 1;

      // Copy the original schedule.
      for (j = 0; j < schedules[i]->NbRows; ++j)
	{
	  ret[i]->p[j + 1][0] = schedules[i]->p[j][0];
	  for (k = 1; k <= schedules[i]->NbRows; ++k)
	    ret[i]->p[j + 1][k + 1] = schedules[i]->p[j][k];
	  for (; k < schedules[i]->NbColumns; ++k)
	    ret[i]->p[j + 1][k + 2] = schedules[i]->p[j][k];
	}
    }

  return ret;
}

CandlProgram*
ls_transformations_tile_program (s_ls_options_t* options,
				 CandlProgram* program,
				 PipMatrix** sched)
{
  int i, j, k;
  int dim_transfo;
  CandlProgram* ret = candl_program_malloc ();
  PipMatrix* domain;
  PipMatrix* bd;
  ret->context = program->context;
  ret->nb_statements = program->nb_statements;
  ret->statement = XMALLOC(CandlStatement*, ret->nb_statements);


  for (i = 0; i < program->nb_statements; ++i)
    {
      ret->statement[i] = candl_statement_malloc ();

      // Copy statement information.
      ret->statement[i]->depth = program->statement[i]->depth;
      ret->statement[i]->label = program->statement[i]->label;
      ret->statement[i]->type = program->statement[i]->type;
      ret->statement[i]->depth = program->statement[i]->depth;
      ret->statement[i]->index = program->statement[i]->index;
      ret->statement[i]->written = program->statement[i]->written;
      ret->statement[i]->read = program->statement[i]->read;

      bd = program->statement[i]->domain;
      // Expand the domain.
      domain = pip_matrix_alloc (bd->NbRows + 2 * sched[i]->NbRows,
				 bd->NbColumns + 1);
      // Copy the original domain.
      for (j = 0; j < bd->NbRows; ++j)
	{
	  domain->p[j][0] = bd->p[j][0];
	  for (k = 1; k < bd->NbColumns; ++k)
	    domain->p[j][k + 1] = bd->p[j][k];
	}
      // Add the 'first' strip-mine inequalities.
      for (; j < bd->NbRows + sched[i]->NbRows; ++j)
	{
	  domain->p[j][0] = 1;
	  domain->p[j][1] = - options->tile_factor;
	  for (k = 1; k < sched[i]->NbColumns - sched[i]->NbRows; ++k)
	    domain->p[j][k + 1] = options->tile_factor *
	      sched[i]->p[j - bd->NbRows][k + sched[i]->NbRows];
	}
      // Add the 'second' strip-mine inequalities.
      for (; j < bd->NbRows + 2 * sched[i]->NbRows; ++j)
	{
	  domain->p[j][0] = 1;
	  domain->p[j][1] = options->tile_factor;
	  for (k = 1; k < sched[i]->NbColumns - sched[i]->NbRows; ++k)
	    domain->p[j][k + 1] = options->tile_factor *
	      (- sched[i]->p[j - bd->NbRows - sched[i]->NbRows]
	      [k + sched[i]->NbRows]);
	  domain->p[j][domain->NbColumns - 1] += options->tile_factor - 1;
	}
      ret->statement[i]->domain = domain;
    }

  return ret;
}


/**
 * \brief Create a CLooG formatted file for a given point in the legal
 * polytope. Multidimensional version.
 *
 *
 */
int
ls_transformations_generate_multi (s_fm_vector_t** draw,
				   CandlProgram* program,
				   s_ls_options_t* options,
				   unsigned count)
{
/*   if (! options->create_schedfiles) */
/*     return; */

  char			str[100];
  char			buff[10];
  FILE*			t_file;
  unsigned		i;
  PipMatrix**		schedules;
  int			ret;

  PipMatrix**		schedules_tiled;
  CandlProgram*		program_tiled;

  PipOptions* pipoptions;
  PipQuast* solution;
  pipoptions = pip_options_init ();
  pipoptions->Simplify = 1;
  pipoptions->Urs_parms = -1;
  pipoptions->Urs_unknowns = -1;

  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buff, "%09d", count);
  strcat(str, buff);
  strcat(str, ".cloog");

  t_file = fopen (str, "w");

  if (options->tile)
    {
/*       printf ("TILING activated\n"); */
      schedules = ls_transfo_to_cloog_multi (program, draw);
      schedules_tiled = ls_transformations_tile_schedules (program, schedules);
      program_tiled =
	ls_transformations_tile_program (options, program, schedules);
      ret = ls_output_file (t_file, program_tiled, schedules_tiled);

      // Be clean.
      for (i = 0; i < program->nb_statements; ++i)
	{
	  pip_matrix_free (schedules[i]);
	  pip_matrix_free (schedules_tiled[i]);
	}
      XFREE(schedules);
      XFREE(schedules_tiled);
      // Partial free of the program.
      for (i = 0; i < program->nb_statements; ++i)
	{
	  solution = pip_solve (program_tiled->statement[i]->domain,
				NULL, -1, pipoptions);
	  if ((solution != NULL) &&
	      ((solution->list != NULL) || (solution->condition != NULL)))
	    {
	      /* 	      printf ("Ok at %d\n", count); */
	    }
	  else
	    {
	      /* 	      printf ("BAD at %d\n", count); */
	    }
	  pip_quast_free (solution);

	  pip_matrix_free (program_tiled->statement[i]->domain);

	  XFREE(program_tiled->statement[i]);
	}
      XFREE(program_tiled->statement);
      XFREE(program_tiled);
    }
  else
    {
      schedules = ls_transfo_to_cloog_multi (program, draw);
      ret = ls_output_file (t_file, program, schedules);
      // Be clean.
      for (i = 0; i < program->nb_statements; ++i)
	pip_matrix_free (schedules[i]);
      XFREE(schedules);
    }

  fclose (t_file);

  return ret;
}



/**
 * \brief Naive recursive polytope explorer.
 *
 *
 */
static
void
ls_polyhedron_explore_rec (s_ls_space_t* space,
                           unsigned idx,
                           s_fm_vector_t* draw,
                           CandlProgram* program,
                           s_ls_options_t* options)
{
  s_fm_solution_t* sol = space->polyhedron;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t i;

  Z_INIT(i);

  fm_solver_compute_min (&lb, sol->solution[idx - 1].positive,
                         draw, idx - 1, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx - 1].negative,
                         draw, idx - 1, FM_MINLEXICO_INT);

  for (Z_ASSIGN(i, lb->num); Z_CMP(i, <=, Ub->num); Z_INC(i, i))
    {
      fm_vector_assign_int_idx (draw, i, idx);
      if (idx == sol->size)
        {
	  ls_transformations_generate (draw, program, options, space->size);
	  space->size++;
        }
      else
        ls_polyhedron_explore_rec (space, idx + 1, draw, program, options);
    }

  Z_CLEAR(i);
}


/**
 * \brief Entry point for the legal transformation space exploration.
 *
 * @comment Former entry point, see explorer.c
 *
 * @see explorer
 *
 */
void
ls_transformations (s_ls_space_t* space,
		    CandlProgram* program,
		    s_ls_options_t* options)
{
  s_fm_vector_t* draw = fm_vector_alloc (space->polyhedron->size + 1);
  unsigned i;

  for (i = 1; i < space->polyhedron->size + 1; ++i)
    fm_vector_assign_int_idx (draw, options->lb, i);

  if (options->verbose)
    fprintf (options->out_file, ".. Legal space polytope exploration\n");

  ls_polyhedron_explore_rec (space, 1, draw, program, options);

  if (options->verbose)
    {
      fprintf (options->out_file, ".. =  Lower bound: %d\n", options->lb);
      fprintf (options->out_file, ".. =  Upper bound: %d\n", options->Ub);
      fprintf (options->out_file, ".. => Number of drawn schedules: %Ld\n",
	       space->size);
      if (options->create_schedfiles)
	fprintf (options->out_file, ".. Files generated in %s\n",
		 options->transfo_dir);
    }
}
