#  /usr/local/bin/bash  
#! /bin/bash

## Some useful options.
#
## Set cloog binary.
cloog="cloog -esp 1 -compilable 1"
## Set letsee binary.
letsee="src/letsee"
## Set the options to provide to letsee.
letsee_opts="-t multi"
### Set directory where transformations are.
transfo_path="transformations"
## Set GCC default options.
gcc_opt="-lm"
## Set directory where results are to be stored.
results_directory="$transfo_path"

# Test if enough parameters were given.
if [ $# -lt 1 ]; then
   echo "Usage: tester.sh lb Ub plb pUb clb cUb \"compile line\" FILE1.candl [FILE2.candl ...]";
   exit 1;
fi

## Retrieve variables from environment. If the variable is unset, use
## default values at the top of this script.
if [ -z "$GCC" ]; then
    GCC=gcc;
fi
if [ -z "$GCC_OPT" ]; then
    GCC_OPT="$gcc_opt";
fi
if [ -z "$CLOOG" ]; then
    CLOOG="$cloog";
fi
if [ -z "$LETSEE" ]; then
    LETSEE="$letsee";
fi
if [ -z "$LETSEE_OPTS" ]; then
    LETSEE_OPTS="$letsee_opts";
fi
if [ -z "$TRANSFO_PATH" ]; then
    TRANSFO_PATH="$transfo_path";
fi
if [ -z "$RESULTS_DIRECTORY" ]; then
    RESULTS_DIRECTORY="$results_directory";
fi

lb="$1";
Ub="$2";
plb="$3";
pUb="$4";
clb="$5";
cUb="$6";
compile_line="$7";
shift 7;

## Loop on each argument.
while [ $# -ne 0 ]; do
    # Create the temporary results files.
    pattern=`basename $1`;
    pattern=${pattern%.candl}
    pattern="$RESULTS_DIRECTORY/letsee.$pattern.XXXXXX"
    out_file=`mktemp ${pattern}`
    plot_file="$out_file.dat"

    FILENAME=${1%.candl};
    
    echo -e "\033[33m=> Testing $FILENAME\033[0m";
    echo -e "\033[33m=  Test parameters: "
    echo -e "   tlb: $lb tUb: $Ub";
    echo -e "   plb: $plb pUb: $pUb";
    echo -e "   clb: $clb cUb: $cUb";
    echo -e "   Compilation line: $compile_line\033[0m";
    echo -e "";
    echo -e "\033[32mStarting transformations generation...\033[0m";

    ## Lanch letsee.
    ##
    t=`make clean-transfo`
    time ./$LETSEE  $LETSEE_OPTS "$1" -a "$lb" -b "$Ub" -c "$plb" -d "$pUb" -e "$clb" -f "$cUb" -v > $out_file
    cat $out_file

    echo -e "\033[32mStarting code generation and execution...\033[0m";

    ## Compile and run the original code.
    ## 
    orig_target=${FILENAME##*/}
    echo -e "\033[33mProcessing original code\033[0m" >> $out_file;
    $compile_line $GCC_OPT "$TRANSFO_PATH/$orig_target.c" -o "$TRANSFO_PATH/$orig_target" &>/dev/null;
    output_orig_prg=`./$TRANSFO_PATH/$orig_target`;
    echo "$output_orig_prg" >> $out_file;
    # Print progress bar.
    echo -n ".";

    ## Iterate on all transformation files generated by Letsee.
    ##
    count=0;
    for i in `find $TRANSFO_PATH -name "transformation*.cloog"`; do
	target=${i%.cloog};
	/bin/echo -e "\033[33mProcessing $target\033[0m" >> $out_file;
	# Generate a C compilable file from the .cloog file.
	CLOOG=$CLOOG src/scripts/cloog2test/cloog2test -sctp "$FILENAME.program" "$i";
        # Compile and run the code.
	$compile_line $GCC_OPT "$target.c" -o "$target" &>/dev/null;
	output_prg=`./$target`;
	if [ $? -ne 0 ]; then
	    echo -e "\033[31m$target has segfault-ed\033[0m";
#	    exit 1;
	fi

	# Get information and fill the GNUPlot data file.
	prg_idx=`echo "$target" | sed "s/.*_\(.*\)/\1/g"`;
	prg_cycles=`echo "$output_prg" | grep Cycles | cut -d : -f 2`;
	echo "$prg_idx" "$prg_cycles" >> $plot_file;
	echo "$output_prg" >> $out_file;
	count=$(($count + 1));
	# Check if there is no difference between program output and
	# original code output.
	## NOTE: Enabling this code fragment must be done with care:
	## due to imprecision of floating point computation, a tiny
	## difference may occur while the code would still be correct.
#	the_diff=`diff ./$target.output ./$TRANSFO_PATH/$orig_target.output`;
#	if ! [ -z "$the_diff" ]; then
#	    echo -e "\033[31m$target has difference in output\033[0m";
#	else
	    rm -f $target.output;
#	fi
	rm -f $target;
	# Print progress bar.
	echo -n ".";
    done

    echo;

    ## Prepare the data files.
    ## 
    ## Add the original code performance to the GNUPlot data file.
    orig_cycles=`echo "$output_orig_prg" | grep Cycles | cut -d : -f 2`;
    orig_idx=$count;
    echo "# Original code is at position $count" >> $plot_file;
    echo "$count" "$orig_cycles" >> $plot_file;

    ## Rank the best transformations.
    ## 
    ## FIXME: Horrible!
    count=1;
    completed=0;
    min="";
    for i in `grep Cycles $out_file | cut -d : -f 2 | sort -n`; do
       res=`grep -C 1 "$i" $out_file | grep "Processing" | \
                                      sed -e "s/Processing//g"`;
       orig=`echo "$res" | grep original`;
       if [ $count -le 10 ] && [ -z "$orig" ]; then
	   if [ -z "$min" ]; then min="$i"; fi;
           echo -e "\033[33m- Ranked $count:\t$res\033[33m\t $i cycles\033[0m";
       fi
       if ! [ -z "$orig" ]; then 
         filen=`basename $FILENAME`;
         echo -e "\033[33m- Ranked $count:\t Original code ($filen)\t\t $i cycles\033[0m";
	 completed=1;
       fi
       count=$(($count + 1));
       # Stop when interesting informations were retrieved.
       if [ "$completed" -eq 1 ] && [ "$count" -gt 10 ]; then 
	   break; 
       fi
    done

     ## Execute the GNUPlot script to generate the charts (in pdf).
     ## 
     min_idx=`grep "$min" "$plot_file" | sed "s/\(.*\) .*/\1/g"`;
     pt=`grep -v "#" "$plot_file" | wc -l`
     pt=$(($pt - 1));
     grep -v "#" "$plot_file" | cut -d ' ' -f 3  | sort -n > $plot_file.tmp
     echo "set term post enh \"Helvetica\" 24; set output \"$out_file.ps\"; \
 	set nokey ; set xlabel \"Transfo.\"; set ylabel \"Cycles (M)\"; \
 	set title \"$orig_target\"; set autoscale; \
 	set label \"Original\" at $orig_idx,$orig_cycles ;\
 	set label \"Best\" at $min_idx,$min ;\
 	plot \"$plot_file.tmp\"" | gnuplot ;
     ps2pdf $out_file.ps $out_file.pdf;
     rm -f $out_file.tmp;
     rm -f $out_file.ps;
     echo "set term post enh \"Helvetica\" 24; set output \"$out_file.ps\"; \
 	set nokey ; set xlabel \"Transfo. ID\"; set ylabel \"Cycles (M)\"; \
 	set title \"$orig_target\"; set autoscale; \
 	set label \"Original\" at $orig_idx,$orig_cycles ;\
        set arrow from 0,$orig_cycles to $pt,$orig_cycles nohead lt -1 lw 1.2;\
 	plot \"$plot_file\"" | gnuplot ;
     ps2pdf $out_file.ps $out_file.raw.pdf;
     rm -f $out_file.ps;

    # Echoing results filename.
    echo -e "\033[33mResults are stored in template $out_file (.dat, .pdf).\033[0m";

#    # Echoing GCC version.
#    echo -e "\033[36mGCC version used: \033[0m";
#    $GCC --version

    echo -e "\033[32mAll done for input $1.\033[0m";
    
    shift 1;
done
