/*
 * matrix_to_past.c: This file is part of the IR-Converter project.
 *
 * IR-Converter: a library to convert PAST to ScopLib
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#if HAVE_CONFIG_H
# include <irconverter/config.h>
#endif

#include <irconverter/common.h>
#include <irconverter/matrix_to_past.h>


s_past_node_t*
irconverter_access_func_to_past (scoplib_matrix_p mat,
				 char** iterators,
				 int nb_iterators,
				 char** parameters,
				 int nb_parameters,
				 char** arrays,
				 int data_is_char)
{
  int i, j, k;

  s_past_node_t* ret = NULL;
  s_past_node_t** addr = &ret;
  s_past_node_t* pred = NULL;

  for (i = 0; i < mat->NbRows; ++i)
    {
      s_past_node_t* temp = NULL;
      s_past_node_t* cur = NULL;

      // Variable symbol.
      if (SCOPVAL_get_si(mat->p[i][0]) != 0)
        {
	  char* varname = arrays[SCOPVAL_get_si(mat->p[i][0]) - 1];
	  s_symbol_t* symvar;
	  if (data_is_char) {
	    symvar = symbol_add_from_char (NULL, varname);
	  }
	  else
	    symvar = symbol_add_from_data (NULL, varname);
	  cur = past_node_variable_create (symvar);
        }

      // Iterators.
      for (j = 0; j < nb_iterators; ++j)
        {
	  int val = SCOPVAL_get_si(mat->p[i][j + 1]);
	  if (val != 0)
            {
	      s_symbol_t* sym;
	      if (data_is_char)
		sym = symbol_add_from_char (NULL, iterators[j]);
	      else
		sym = symbol_add_from_data (NULL, iterators[j]);
	      s_past_node_t* var = past_node_variable_create (sym);
	      if (val != 1)
                {
		  s_past_node_t* value = past_node_value_create_from_int (val);
		  var = past_node_binary_create (past_mul, value, var);
                }
	      if (temp == NULL)
		temp = var;
	      else
		temp = past_node_binary_create (past_add, temp, var);
            }
        }

      // Parameters.
      for (k = 0; k < nb_parameters; ++k)
        {
	  int val = SCOPVAL_get_si(mat->p[i][j + 1 + k]);
	  if (val != 0)
            {
	      s_symbol_t* sym;
	      if (data_is_char)
		sym = symbol_add_from_char (NULL, parameters[k]);
	      else
		sym = symbol_add_from_data (NULL, parameters[j]);
	      s_past_node_t* var = past_node_variable_create (sym);
	      if (val != 1)
                {
		  s_past_node_t* value = past_node_value_create_from_int (val);
		  var = past_node_binary_create (past_mul, value, var);
                }
	      if (temp == NULL)
		temp = var;
	      else
		temp = past_node_binary_create (past_add, temp, var);
            }
        }

      // Constant.
      int val = SCOPVAL_get_si(mat->p[i][j + 1 + k]);
      if (val != 0)
        {
	  s_past_node_t* value = past_node_value_create_from_int (val);
	  if (temp == NULL)
	    temp = value;
	  else
	    temp = past_node_binary_create (past_add, temp, value);
        }

      // Array reference.
      if (temp || (i < mat->NbRows - 1 && SCOPVAL_get_si(mat->p[i + 1][0]) == 0)
	  || past_node_is_a (cur, past_arrayref))
        {
	  if (temp == NULL) {
	    temp = past_node_value_create_from_int (0);
	  }
	  if (!cur) {
	    // Continuing an existing array
	    pred = past_node_binary_create(past_arrayref, pred, temp);
	  } else {
	    // Starting a new array ref
	    if (pred) {
	      *addr = pred;
	      addr = &(*addr)->next;
	    }
	    pred = past_node_binary_create (past_arrayref, cur, temp);
	  }
        } else {
	// Not an array
	if (pred) {
	  *addr = pred;
	  addr = &(*addr)->next;
	}
	pred = cur;
      }
    }

  // Chain last access
  if (pred)
    *addr = pred;

  return ret;
}
