#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include <piplib/piplib.h>
#include <piplib-gmp/piplib-gmp.h>

/**
 * pip_hybrid_solve:
 * Use long long int for the solver. If integer overflow occurs,
 * promote to piplib-gmp and re-run the solver.
 *
 * If no integer overflow occurs, *solquast is set to the PipQuast*
 * solution quast (possibly NULL) and *mpsolquast is set to NULL.
 * If integer overflow occurs, *solquast is set to the PipMPQuast*
 * solution quast (possibly NULL), and *solquast is set to NULL.
 *
 */
void
piplib_hybrid_solve (PipMatrix* domain,
		     PipMatrix* context,
		     int Bg,
		     PipOptions* options,
		     PipQuast** solquast,
		     void** mpsolquast)
{
  if (domain == NULL)
    return;
  PipQuast* solution = NULL;
  int has_overflow = pip_solve_core (domain, context, -1, options, &solution);
  if (! has_overflow)
    {
      *solquast = solution;
      *mpsolquast = NULL;
    }
  else
    {
      int i, j;
      PipMPMatrix* mpdomain = pipmp_matrix_alloc (domain->NbRows,
						  domain->NbColumns);
      for (i = 0; i < domain->NbRows; ++i)
	for (j = 0; j < domain->NbColumns; ++j)
	  mpzvalue_set_si(mpdomain->p[i][j], VALUE_TO_INT(domain->p[i][j]));
      PipMPMatrix* mpcontext = NULL;
      if (context)
	{
	  mpcontext = pipmp_matrix_alloc (context->NbRows,
					  context->NbColumns);
	  for (i = 0; i < context->NbRows; ++i)
	    for (j = 0; j < context->NbColumns; ++j)
	      mpzvalue_set_si(mpcontext->p[i][j],
			      VALUE_TO_INT(context->p[i][j]));
	}
      PipMPOptions* mpoptions = pipmp_options_init();
      mpoptions->Simplify = options->Simplify;
      mpoptions->Urs_parms = options->Urs_parms;
      mpoptions->Urs_unknowns = options->Urs_unknowns;
      mpoptions->Nq = options->Nq;
      mpoptions->Verbose = options->Verbose;
      mpoptions->Deepest_cut = options->Deepest_cut;
      mpoptions->Maximize = options->Maximize;
      PipMPQuast* mpsolution =
	pipmp_solve (mpdomain, mpcontext, -1, mpoptions);
      pipmp_options_free (mpoptions);
      pipmp_matrix_free (mpdomain);
      pipmp_matrix_free (mpcontext);
      *solquast = NULL;
      *mpsolquast = mpsolution;
    }
}

